<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use ActiveCollab\Cookies\CookiesInterface;
use ActiveCollab\DatabaseConnection\Connection\MysqliConnection as DatabaseConnection;
use ActiveCollab\Encryptor\Encryptor;
use ActiveCollab\Encryptor\EncryptorInterface;
use ActiveCollab\JobsQueue\Dispatcher as JobsDispatcher;
use ActiveCollab\JobsQueue\Queue\MySqlQueue as MySqlJobsQueue;
use ActiveCollab\Logger\AppRequest\CliRequest;
use ActiveCollab\Logger\AppRequest\HttpRequest;
use ActiveCollab\Logger\AppResponse\HttpResponse;
use ActiveCollab\Logger\ErrorHandler\ErrorHandler;
use ActiveCollab\Logger\ErrorHandler\ErrorHandlerInterface;
use ActiveCollab\Logger\Factory\Factory as LoggerFactory;
use Angie\Authentication;
use Angie\Authentication\AuthorizationIntegrationLocator\AuthorizationIntegrationLocator;
use Angie\AutoUpgrade;
use Angie\Error;
use Angie\Events;
use Angie\Globalization;
use Angie\Http\Request;
use Angie\Http\RequestFactory;
use Angie\Http\RequestHandler\RequestHandler;
use Angie\Inflector;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Angie appliction interface.
 *
 * @package angie.library.application
 */
final class AngieApplication
{
    // Application modes
    const IN_DEVELOPMENT = 'development';
    const IN_DEBUG_MODE = 'debug';
    const IN_PRODUCTION = 'production';

    const STABLE_CHANNEL = 0;
    const BETA_CHANNEL = 1;
    const EDGE_CHANNEL = 2;

    // Api token variable name
    const API_TOKEN_HEADER_NAME = 'HTTP_X_ANGIE_AUTHAPITOKEN';

    // ---------------------------------------------------
    //  Meta information
    // ---------------------------------------------------

    /**
     * Return application name.
     *
     * @param  bool   $verbose
     * @return string
     */
    public static function getName($verbose = false)
    {
        return $verbose ? 'Active Collab' : 'ActiveCollab';
    }

    /**
     * Return application name.
     *
     * @return string
     */
    public static function getUrl()
    {
        return 'https://www.activecollab.com/index.html';
    }

    /**
     * Return application version.
     *
     * @return string
     */
    public static function getVersion()
    {
        return APPLICATION_VERSION;
    }

    /**
     * Return build number.
     *
     * @return string
     */
    public static function getBuild()
    {
        return APPLICATION_BUILD == '%APPLICATION-BUILD%' ? 'DEV' : APPLICATION_BUILD;
    }

    /**
     * Return vendor name.
     */
    public static function getVendor()
    {
        return 'A51';
    }

    /**
     * Return license key.
     *
     * @return string
     */
    public static function getLicenseKey()
    {
        return LICENSE_KEY;
    }

    /**
     * Cached account ID.
     *
     * @var int
     */
    private static $account_id = false;

    /**
     * Return account ID.
     *
     * @return int
     * @throws InvalidParamError
     */
    public static function getAccountId()
    {
        if (self::$account_id === false) {
            if (self::isOnDemand()) {
                self::$account_id = ON_DEMAND_INSTANCE_ID;
            } else {
                self::$account_id = (int) explode('/', self::getLicenseKey())[1];
            }

            if (empty(self::$account_id) && defined('ANGIE_IN_TEST') && ANGIE_IN_TEST) {
                self::$account_id = 145040;
            }
        }

        return self::$account_id;
    }

    /**
     * Return license agreement URL.
     *
     * @return string
     */
    public static function getLicenseAgreementUrl()
    {
        return 'https://activecollab.com/help/books/licensing/self-hosted-license-agreement.html';
    }

    /**
     * Return anonymous usage stats.
     *
     * @param  DateValue|null $date
     * @return array|bool
     */
    public static function getStats(DateValue $date = null)
    {
        if (empty($date)) {
            $date = DateValue::now();
        }

        $stats = [];

        Events::trigger('on_extra_stats', [&$stats, $date]);

        if (self::isOnDemand()) {
            OnDemand::enrichStats($stats, $date);
        }

        return $stats;
    }

    /**
     * Return storage usage.
     *
     * @param  bool           $use_cache
     * @param  DateValue|null $date
     * @return int
     */
    public static function getStorageUsage($use_cache = false, DateValue $date = null)
    {
        if (empty($date)) {
            $date = DateValue::now();
        }

        return self::cache()->get('storage_usage_' . $date->format('Y-m-d'), function () use ($date) {
            $storage_usage = 0;
            Angie\Events::trigger('on_calculate_storage_usage', [&$storage_usage, $date]);

            return $storage_usage;
        }, !$use_cache);
    }

    // ---------------------------------------------------
    //  Bootstrapping
    // ---------------------------------------------------

    /**
     * Load system so it can properly handle HTTP request.
     */
    public static function bootstrapForHttpRequest()
    {
        self::initEnvironment();
        self::initErrorHandler();

        if (!self::isInstalled()) {
            self::initInstaller();

            return;
        }

        self::initCache();
        self::initDatabaseConnection();
        self::initFrameworks();
        self::initFirewall();
        self::initModules();
        self::initRouter();
        self::initEventsManager();
    }

    /**
     * Returns true if $version is a valid angie application version number.
     *
     * @param  string $version
     * @return bool
     */
    public static function isValidVersionNumber($version)
    {
        if (strpos($version, '.') !== false) {
            $parts = explode('.', $version);

            if (count($parts) == 3) {
                foreach ($parts as $part) {
                    if (!is_numeric($part)) {
                        return false;
                    }
                }

                return true;
            }
        }

        return false;
    }

    /**
     * Load system so it can properly handle CLI request (scheduled task etc).
     *
     * @param bool $init_router
     * @param bool $init_events
     * @param bool $init_modules
     */
    public static function bootstrapForCommandLineRequest($init_router = false, $init_events = true, $init_modules = true)
    {
        self::initEnvironment();
        self::initErrorHandler();

        self::log()->setAppRequest(new CliRequest(self::getAccountId(), $_SERVER['argv']));

        self::initCache();
        self::initDatabaseConnection();
        self::initFrameworks();

        if ($init_modules) {
            self::initModules();
        }

        if ($init_router) {
            self::initRouter();
        }

        if ($init_events) {
            self::initEventsManager();
        }
    }

    /**
     * Bootstrap for installer.
     */
    public static function bootstrapForInstallation()
    {
        self::initEnvironment(false);
    }

    /**
     * Bootstrap system to run automated tests.
     */
    public static function bootstrapForTest()
    {
        self::initEnvironment();
        self::initCache(true);
        self::initDatabaseConnection();
        self::initModel('test');
        self::initFrameworks();
        self::initModules();
        self::initRouter();
        self::initEventsManager();
    }

    /**
     * Initialize PHP environment.
     *
     * @param bool $register_shutdown_function
     */
    public static function initEnvironment($register_shutdown_function = true)
    {
        // CLI can start the session earlier, let's avoid warnings
        if (session_status() != PHP_SESSION_ACTIVE) {
            session_start();
        }

        set_include_path('');
        error_reporting(E_ALL);

        ini_set('display_errors', self::isInProduction() ? 0 : 1);

        if ($register_shutdown_function) {
            register_shutdown_function(['AngieApplication', 'shutdown']);
        }
    }

    /**
     * Initialize HTTP environment.
     *
     * @param Request $request
     */
    public static function initHttpEnvironment(Request $request)
    {
        $http_accept_encoding = $request->getServerParam('HTTP_ACCEPT_ENCODING');
        self::$accepts_gzip = $http_accept_encoding && strpos($http_accept_encoding, 'gzip') !== false;
    }

    /**
     * @var ErrorHandlerInterface
     */
    private static $error_handler;

    /**
     * Init error handler.
     */
    public static function initErrorHandler()
    {
        if (empty(self::$error_handler)) {
            self::$error_handler = (new ErrorHandler(self::log()))
                ->setHowToHandleError(E_STRICT, ErrorHandlerInterface::SILENCE)
                ->initialize();
        }
    }

    /**
     * Initialize caching service.
     *
     * @param bool $clear
     */
    public static function initCache($clear = false)
    {
        self::cache()->setLifetime(CACHE_LIFETIME);

        if ($clear) {
            self::cache()->clear();
        }
    }

    /**
     * Initialize database connection.
     */
    public static function initDatabaseConnection()
    {
        try {
            DB::setConnection('default', new MySQLDBConnection(DB_HOST, DB_USER, DB_PASS, DB_NAME));
        } catch (Exception $e) {
            if (!self::isInProduction()) {
                throw $e;
            }

            trigger_error('Failed to connect to database');
        }
    }

    /**
     * @var Smarty
     */
    private static $smarty;

    /**
     * @return Smarty
     */
    public static function &getSmarty()
    {
        if (empty(self::$smarty)) {
            self::$smarty = new Smarty();

            self::$smarty->setCompileDir(COMPILE_PATH);
            self::$smarty->setCacheDir(ENVIRONMENT_PATH . '/cache');
            self::$smarty->compile_check = true;
            self::$smarty->registerFilter('variable', 'clean'); // {$foo nofilter}
        }

        return self::$smarty;
    }

    /**
     * Initialize application model.
     *
     * @param string $environment
     */
    public static function initModel($environment = null)
    {
        if (AngieApplicationModel::isEmpty()) {
            AngieApplicationModel::load(self::getFrameworkNames(), self::getModuleNames());
        }

        AngieApplicationModel::drop();
        AngieApplicationModel::init($environment);
    }

    /**
     * Array of loaded frameworks and modules.
     *
     * @var AngieFramework[]|AngieModule[]
     */
    private static $loaded_frameworks_and_modules = [];

    /**
     * Flag that is set to true when frameworks and frameworks are initialized.
     *
     * @var bool
     */
    private static $frameworks_initialized = false, $modules_initialized = false;

    /**
     * Initialize application frameworks.
     */
    public static function initFrameworks()
    {
        if (!empty(self::$frameworks_initialized)) {
            return;
        }

        foreach (self::getFrameworks() as $framework) {
            self::$loaded_frameworks_and_modules[$framework->getName()] = $framework; // Set as loaded before we call init.php

            $framework->init();

            self::getSmarty()->addPluginsDir($framework->getPath() . '/helpers');
        }

        self::$frameworks_initialized = true;
    }

    /**
     * Initialize installed application modules.
     */
    public static function initModules()
    {
        if (!empty(self::$modules_initialized)) {
            return;
        }

        foreach (self::getModules() as $module) {
            self::$loaded_frameworks_and_modules[$module->getName()] = $module; // Set as loaded before we call init.php

            $module->init();

            self::getSmarty()->addPluginsDir($module->getPath() . '/helpers');
        }

        self::$modules_initialized = true;
    }

    /**
     * Return list of frameworks that this application users.
     *
     * @return array
     */
    public static function getFrameworkNames()
    {
        return [
            'environment',
            'activity_logs',
            'history',
            'email',
            'attachments',
            'notifications',
            'subscriptions',
            'comments',
            'categories',
            'labels',
            'payments',
            'reminders',
            'calendars',
        ];
    }

    /**
     * Return list of modules that this application ships with.
     *
     * @return array
     */
    public static function getModuleNames()
    {
        $result = ['system', 'discussions', 'files', 'invoicing', 'tasks', 'notes', 'tracking'];

        if (self::isOnDemand()) {
            $result[] = 'on_demand';
        }

        return $result;
    }

    /**
     * Initialize route.
     */
    public static function initRouter()
    {
        Router::init(self::$frameworks, self::$modules, !self::isOnDemand());
    }

    /**
     * Init events manager.
     */
    public static function initEventsManager()
    {
        foreach (self::$frameworks as $framework) {
            $framework->defineHandlers();
        }

        foreach (self::$modules as $module) {
            $module->defineHandlers();
        }
    }

    /**
     * Include core installer files.
     */
    public static function includeCoreInstallerFiles()
    {
        require_once ANGIE_PATH . '/classes/application/installer/AngieApplicationInstaller.class.php';
        require_once ANGIE_PATH . '/classes/application/installer/AngieApplicationInstallerAdapter.class.php';
    }

    /**
     * Initialize installer.
     *
     * @param string $adapter_class
     * @param string $adapter_class_path
     */
    public static function initInstaller($adapter_class = null, $adapter_class_path = null)
    {
        self::includeCoreInstallerFiles();
        AngieApplicationInstaller::init($adapter_class, $adapter_class_path);
    }

    /**
     * Initialize firewall.
     */
    public static function initFirewall()
    {
        self::firewall()->initialize();
    }

    // ---------------------------------------------------
    //  Delegates
    // ---------------------------------------------------

    /**
     * Place where we'll keep delegate instances.
     *
     * @var array
     */
    private static $delegate_instances = [];

    /**
     * Return a particular delegate.
     *
     * @param $delegate
     * @return AngieDelegate
     * @throws InvalidParamError
     * @throws InvalidInstanceError
     */
    private static function &getDelegate($delegate)
    {
        if (!isset(self::$delegate_instances[$delegate]) || !(self::$delegate_instances[$delegate] instanceof AngieDelegate)) {
            $delegate_class = 'Angie' . Inflector::camelize($delegate) . 'Delegate';

            if (!class_exists($delegate_class, true)) {
                throw new InvalidParamError('delegate', $delegate, "Delegate '$delegate' not found");
            }

            $delegate_instance = new $delegate_class();
            if ($delegate_instance instanceof AngieDelegate) {
                self::$delegate_instances[$delegate] = $delegate_instance;
            } else {
                throw new InvalidInstanceError('delegate_instance', $delegate_instance, 'AngieDelegate');
            }
        }

        return self::$delegate_instances[$delegate];
    }

    /**
     * Return cache delegate instance.
     *
     * @return AngieDelegate|AngieCacheDelegate
     */
    public static function &cache()
    {
        return self::getDelegate('cache');
    }

    /**
     * Return launcher delegate instance.
     *
     * @return AngieDelegate|AngieLauncherDelegate
     */
    public static function &launcher()
    {
        return self::getDelegate('launcher');
    }

    /**
     * Return notifications delegate instance.
     *
     * @return AngieDelegate|AngieNotificationsDelegate
     */
    public static function &notifications()
    {
        return self::getDelegate('notifications');
    }

    /**
     * Return migration delegate.
     *
     * @return AngieDelegate|AngieMigrationDelegate
     */
    public static function &migration()
    {
        return self::getDelegate('migration');
    }

    /**
     * Return firewall delegate instance.
     *
     * @return AngieDelegate|AngieFirewallDelegate
     */
    public static function &firewall()
    {
        return self::getDelegate('firewall');
    }

    /**
     * @var JobsDispatcher
     */
    private static $jobs_dispatcher;

    /**
     * Connection to global jobs queue. It is closed on shutdown (AngieApplication::shutdown()).
     *
     * @var MySQLi
     */
    private static $global_job_queue_connection;

    /**
     * Interface to jobs dispatcher.
     *
     * @return JobsDispatcher
     */
    public static function &jobs()
    {
        if (empty(self::$jobs_dispatcher)) {
            // @TODO: Remove when launched
            if (defined('IS_NEW_SHEPHERD') && IS_NEW_SHEPHERD &&
                defined('ACTIVECOLLAB_JOB_CONSUMER_MYSQL_HOST') && defined('ACTIVECOLLAB_JOB_CONSUMER_MYSQL_USER') &&
                defined('ACTIVECOLLAB_JOB_CONSUMER_MYSQL_PASS') && defined('ACTIVECOLLAB_JOB_CONSUMER_MYSQL_NAME')) {
                self::$global_job_queue_connection = new MySQLi(
                    ACTIVECOLLAB_JOB_CONSUMER_MYSQL_HOST,
                    ACTIVECOLLAB_JOB_CONSUMER_MYSQL_USER,
                    ACTIVECOLLAB_JOB_CONSUMER_MYSQL_PASS,
                    ACTIVECOLLAB_JOB_CONSUMER_MYSQL_NAME
                );

                if (self::$global_job_queue_connection->connect_error) {
                    throw new RuntimeException('Failed to connect to database. MySQL said: ' . self::$global_job_queue_connection->connect_error);
                }

                self::$global_job_queue_connection->query('SET NAMES utf8mb4');

                $connection = new DatabaseConnection(self::$global_job_queue_connection);
            } elseif (defined('GLOBAL_JOBS_QUEUE_HOST') && defined('GLOBAL_JOBS_QUEUE_USER') && defined('GLOBAL_JOBS_QUEUE_PASS') && defined('GLOBAL_JOBS_QUEUE_NAME')) {
                self::$global_job_queue_connection = new MySQLi(GLOBAL_JOBS_QUEUE_HOST, GLOBAL_JOBS_QUEUE_USER, GLOBAL_JOBS_QUEUE_PASS, GLOBAL_JOBS_QUEUE_NAME);

                if (self::$global_job_queue_connection->connect_error) {
                    throw new RuntimeException('Failed to connect to database. MySQL said: ' . self::$global_job_queue_connection->connect_error);
                }

                self::$global_job_queue_connection->query('SET NAMES utf8mb4');

                $connection = new DatabaseConnection(self::$global_job_queue_connection);
            } else {
                $connection = new DatabaseConnection(DB::getConnection()->getLink());
            }

            $mysql_queue = new MySqlJobsQueue($connection, false);
            $mysql_queue->extractPropertyToField('instance_id');

            self::$jobs_dispatcher = new JobsDispatcher($mysql_queue);
            self::$jobs_dispatcher->registerChannels(
                SystemModule::MAINTENANCE_JOBS_QUEUE_CHANNEL,
                EmailIntegration::JOBS_QUEUE_CHANNEL,
                SearchIntegration::JOBS_QUEUE_CHANNEL,
                WebhooksIntegration::JOBS_QUEUE_CHANNEL
            );
        }

        return self::$jobs_dispatcher;
    }

    /**
     * Return a connection that is connected to jobs queue.
     *
     * @return DatabaseConnection
     */
    public static function jobsConnection()
    {
        self::jobs(); // Make sure that we open a connection

        if (defined('IS_NEW_SHEPHERD') && IS_NEW_SHEPHERD &&
            defined('ACTIVECOLLAB_JOB_CONSUMER_MYSQL_HOST') && defined('ACTIVECOLLAB_JOB_CONSUMER_MYSQL_USER') &&
            defined('ACTIVECOLLAB_JOB_CONSUMER_MYSQL_PASS') && defined('ACTIVECOLLAB_JOB_CONSUMER_MYSQL_NAME')) {
            return new DatabaseConnection(self::$global_job_queue_connection);
        }

        if (defined('GLOBAL_JOBS_QUEUE_HOST') && defined('GLOBAL_JOBS_QUEUE_USER') && defined('GLOBAL_JOBS_QUEUE_PASS') && defined('GLOBAL_JOBS_QUEUE_NAME')) {
            return new DatabaseConnection(self::$global_job_queue_connection);
        }

        return new DatabaseConnection(DB::getConnection()->getLink());
    }

    /**
     * Return memories delegate.
     *
     * @return AngieDelegate|AngieMemoriesDelegate
     */
    public static function &memories()
    {
        return self::getDelegate('memories');
    }

    /**
     * @var AutoUpgrade
     */
    private static $auto_upgrade;

    /**
     * Return auto-upgrade instance.
     *
     * @return AutoUpgrade
     */
    public static function &autoUpgrade()
    {
        if (empty(self::$auto_upgrade)) {
            self::$auto_upgrade = new AutoUpgrade(self::memories()->getInstance());
        }

        return self::$auto_upgrade;
    }

    /**
     * @var Authentication
     */
    private static $authentication;

    /**
     * @return Authentication
     */
    public static function &authentication()
    {
        if (empty(self::$authentication)) {
            $is_on_demand_next_gen = defined('IS_NEW_SHEPHERD') && IS_NEW_SHEPHERD;
            $authorization_locator = new AuthorizationIntegrationLocator(self::isOnDemand(), $is_on_demand_next_gen, (string) ConfigOptions::getValue('authorization_integration'));

            /** @var AuthorizationIntegrationInterface $authorization_integration */
            $authorization_integration = $authorization_locator->getAuthorizationIntegration();

            self::$authentication = new Angie\Authentication($authorization_integration);

            self::$authentication->setOnAuthenciatedUserChanged(function (User $user = null) {
                if ($user) {
                    Globalization::setCurrentLocaleByUser($user);
                }
            });
        }

        return self::$authentication;
    }

    /**
     * Return true if authentication is loaded.
     *
     * @return bool
     */
    public static function isAuthenticationLoaded()
    {
        return !empty(self::$authentication);
    }

    /**
     * Reset authentication service.
     *
     * This method is used for testing only, so we can reset auth layer between tests
     */
    public static function unsetAuthentication()
    {
        self::$authentication = null;
    }

    /**
     * @var EncryptorInterface
     */
    private static $encryptor;

    /**
     * Return pre-configured encryptor.
     *
     * @return EncryptorInterface
     */
    public static function &encryptor()
    {
        if (empty(self::$encryptor)) {
            self::$encryptor = new Encryptor(APPLICATION_UNIQUE_KEY);
        }

        return self::$encryptor;
    }

    /**
     * @var CookiesInterface|null
     */
    private static $cookies;

    /**
     * @return CookiesInterface
     */
    public static function &cookies()
    {
        if (empty(self::$cookies)) {
            $bits = parse_url(ROOT_URL);

            $cookies_host = empty($bits['host']) || in_array($bits['host'], ['localhost', '0.0.0.0', '127.0.0.1', 'activecollab.dev']) ? '' : $bits['host'];
            $cookies_path = empty($bits['path']) ? '/' : $bits['path'];
            $cookies_secure = isset($bits['scheme']) && $bits['scheme'] == 'https';

            self::$cookies = (new \ActiveCollab\Cookies\Cookies(new \ActiveCollab\Cookies\Adapter\Adapter(), new \Angie\Utils\CurrentTimestamp()))
                ->prefix('activecollab_')
                ->domain($cookies_host)
                ->path($cookies_path)
                ->secure($cookies_secure)
                ->encryptor(self::encryptor());
        }

        return self::$cookies;
    }

    /**
     * @return string
     */
    public static function getSessionIdCookieName()
    {
        return 'us_for_' . sha1(ROOT_URL);
    }

    /**
     * @return string
     */
    public static function getCsrfValidatorCookieName()
    {
        return 'csrf_validator_for_' . sha1(ROOT_URL);
    }

    /**
     * @return string
     */
    public static function getLanguageCookieName()
    {
        return 'ul_for_' . sha1(ROOT_URL);
    }

    /**
     * @var \ActiveCollab\Logger\LoggerInterface
     */
    private static $logger;

    /**
     * Return logger instance.
     *
     * @return \ActiveCollab\Logger\LoggerInterface
     */
    public static function &log()
    {
        if (empty(self::$logger)) {
            $factory = new LoggerFactory();
            $factory->setAdditionalEnvArguments(['account_id' => self::getAccountId()]);
            $factory->addExceptionSerializer(function ($argument_name, $exception, array &$context) {
                if ($exception instanceof \Angie\Error) {
                    foreach ($exception->getParams() as $k => $v) {
                        $context["{$argument_name}_param_{$k}"] = $v;
                    }
                }
            });

            $environment = 'production';
            $logger_type = \ActiveCollab\Logger\LoggerInterface::BLACKHOLE;
            $logger_arguments = [];

            if (!defined('ANGIE_IN_TEST') || !ANGIE_IN_TEST) {
                if (self::isOnDemand()) {
                    if (self::isEdgeChannel()) {
                        $environment = 'staging';
                    }

                    if (defined('GRAYLOG_HOST') && defined('GRAYLOG_PORT')) {
                        $logger_type = \ActiveCollab\Logger\LoggerInterface::GRAYLOG;
                        $logger_arguments = [GRAYLOG_HOST, GRAYLOG_PORT];
                    }
                } else {
                    if (self::isInDevelopment()) {
                        $environment = 'development';
                    }

                    if (!self::isInProduction()) {
                        $logger_type = \ActiveCollab\Logger\LoggerInterface::FILE;
                        $logger_arguments = [ENVIRONMENT_PATH . '/logs', 'log.txt', 0777];
                    }
                }
            }

            $log_level = $environment === 'production' ? \ActiveCollab\Logger\LoggerInterface::LOG_FOR_PRODUCTION : \ActiveCollab\Logger\LoggerInterface::LOG_FOR_DEBUG;

            self::$logger = $factory->create(self::getName(), self::getVersion(), $environment, $log_level, $logger_type, ...$logger_arguments);
        }

        return self::$logger;
    }

    /**
     * @var AngieStorageDelegate
     */
    private static $storage;

    /**
     * Return storage instance.
     *
     * @return AngieStorageDelegate
     */
    public static function &storage()
    {
        if (!self::$storage) {
            self::$storage = new AngieStorageDelegate();
        }

        return self::$storage;
    }

    /**
     * Unset storage instance.
     *
     * This is used because storage delegate is static instead of instance. @TODO It should be removed as it's technical debt.
     */
    public static function unsetStorageDelegate()
    {
        self::$storage = null;
    }

    // ---------------------------------------------------
    //  Application Mode
    // ---------------------------------------------------

    /**
     * Returns true if application is in development mode.
     *
     * @return bool
     */
    public static function isInDevelopment()
    {
        return defined('APPLICATION_MODE') ? APPLICATION_MODE == self::IN_DEVELOPMENT : true;
    }

    /**
     * Returns true if application is in debug mode.
     *
     * @return bool
     */
    public static function isInDebugMode()
    {
        return defined('APPLICATION_MODE') && APPLICATION_MODE == self::IN_DEBUG_MODE;
    }

    /**
     * Returns true if application is in production mode.
     *
     * @return bool
     */
    public static function isInProduction()
    {
        return defined('APPLICATION_MODE') && APPLICATION_MODE == self::IN_PRODUCTION;
    }

    /**
     * Returns true if application is in on demand mode.
     *
     * @return bool
     */
    public static function isOnDemand()
    {
        return defined('IS_ON_DEMAND') && IS_ON_DEMAND;
    }

    /**
     * Return true if we have a paid on demand account here.
     *
     * @return bool
     */
    public static function isPaidOnDemand()
    {
        return self::isOnDemand() && OnDemand::isPaid();
    }

    /**
     * Return true if this on demand instance is on the edge deployment channel.
     *
     * @return bool
     */
    public static function isEdgeChannel()
    {
        return (self::isInDevelopment() || self::isOnDemand()) && defined('ON_DEMAND_APPLICATION_CHANNEL') && ON_DEMAND_APPLICATION_CHANNEL === self::EDGE_CHANNEL;
    }

    /**
     * Return true if this on demand instance is on the beta deployment channel.
     *
     * @return bool
     */
    public static function isBetaChannel()
    {
        return (self::isInDevelopment() || self::isOnDemand()) && defined('ON_DEMAND_APPLICATION_CHANNEL') && ON_DEMAND_APPLICATION_CHANNEL === self::BETA_CHANNEL;
    }

    /**
     * Return true if this on demand instance is on the stable deployment channel.
     *
     * @return bool
     */
    public static function isStableChannel()
    {
        return !self::isBetaChannel() && !self::isEdgeChannel();
    }

    /**
     * Return deployment channel.
     *
     * @return int
     */
    public static function getDeploymentChannel()
    {
        if (self::isEdgeChannel()) {
            return self::EDGE_CHANNEL;
        }

        if (self::isBetaChannel()) {
            return self::BETA_CHANNEL;
        }

        return self::STABLE_CHANNEL;
    }

    // ---------------------------------------------------
    //  Request Handling
    // ---------------------------------------------------

    /**
     * Handle HTTP request.
     */
    public static function handleHttpRequest()
    {
        $is_in_test = defined('ANGIE_IN_TEST') && ANGIE_IN_TEST;

        if (php_sapi_name() === 'cli' && !$is_in_test) {
            throw new RuntimeException('HTTP request handler is available to CLI only for testing');
        }

        $request = (new RequestFactory())->createFromGlobals();
        self::log()->setAppRequest(new HttpRequest($request));

        $response = new \Angie\Http\Response();
        $response = self::executeHttpMiddlewareStack($request, $response);
        self::log()->setAppResponse(new HttpResponse($response));

        (new \Zend\Diactoros\Response\SapiEmitter())->emit($response);

        die();
    }

    /**
     * @param  ServerRequestInterface $request
     * @param  ResponseInterface      $response
     * @return ResponseInterface
     */
    public static function executeHttpMiddlewareStack(ServerRequestInterface $request, ResponseInterface $response)
    {
        self::log()->setAppRequest(new \ActiveCollab\Logger\AppRequest\HttpRequest($request));
        self::getClientAcceptsGzipFromRequest($request);

        $request_handler = new RequestHandler(
            self::authentication(),
            self::cookies(),
            self::encryptor(),
            function ($controller_name, $module_name) {
                self::useController($module_name, $controller_name);
            },
            self::isInDebugMode() || self::isInProduction(),
            self::log()
        );

        return $request_handler->handleRequest($request, $response);
    }

    /**
     * True if client can accept zipped content.
     *
     * @var bool
     */
    private static $accepts_gzip = false;

    /**
     * Returns true if client accepts GZip.
     *
     * @return bool
     */
    public static function clientAcceptsGzip()
    {
        return self::$accepts_gzip;
    }

    /**
     * Get (and cache) info wether client accepts gzip encoded response.
     *
     * @param ServerRequestInterface $request
     */
    private static function getClientAcceptsGzipFromRequest(ServerRequestInterface $request)
    {
        $server_params = $request->getServerParams();

        $http_accept_encoding = array_var($server_params, 'HTTP_ACCEPT_ENCODING');
        self::$accepts_gzip = $http_accept_encoding && strpos($http_accept_encoding, 'gzip') !== false;
    }

    /**
     * Return user IP address.
     *
     * @return string
     */
    public static function getVisitorIp()
    {
        return array_var($_SERVER, 'REMOTE_ADDR', '127.0.0.1');
    }

    /**
     * Return visitor's user agent string.
     *
     * @return string
     */
    public static function getVisitorUserAgent()
    {
        return isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : null;
    }

    /**
     * Return request schema (http:// or https://).
     *
     * @return string
     */
    public static function getRequestSchema()
    {
        return ((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || (isset($_SERVER['SERVER_PORT']) && $_SERVER['SERVER_PORT'] == 443) || (isset($_SERVER['HTTP_X_REAL_PORT']) && $_SERVER['HTTP_X_REAL_PORT'] == 443)) ? 'https://' : 'http://';
    }

    /**
     * Invalidate initial settings timestamp.
     */
    public static function invalidateInitialSettingsCache()
    {
        ConfigOptions::setValue('initial_settings_timestamp', time());
    }

    // ---------------------------------------------------
    //  Frameworks and modules
    // ---------------------------------------------------

    /**
     * Cached array of application frameworks.
     *
     * @var AngieFramework[]
     */
    private static $frameworks = false;

    /**
     * Return list of available application frameworks.
     *
     * @return AngieFramework[]
     * @throws FileDnxError
     * @throws ClassNotImplementedError
     */
    public static function &getFrameworks()
    {
        if (self::$frameworks === false) {
            self::$frameworks = [];

            foreach (self::getFrameworkNames() as $framework_name) {
                $framework_class = Inflector::camelize($framework_name) . 'Framework';

                require_once ANGIE_PATH . "/frameworks/$framework_name/$framework_class.php";

                $framework = new $framework_class();
                if ($framework instanceof AngieFramework) {
                    self::$frameworks[] = $framework;
                } else {
                    throw new ClassNotImplementedError($framework_class, ANGIE_PATH . "/frameworks/$framework_name/$framework_class.php", 'Framwork definition class not found');
                }
            }
        }

        return self::$frameworks;
    }

    /**
     * Cached list of installed modules.
     *
     * @var AngieModule[]
     */
    private static $modules = false;

    /**
     * Return list of installed application modules.
     *
     * @return AngieModule[]
     * @throws FileDnxError
     * @throws ClassNotImplementedError
     */
    public static function &getModules()
    {
        if (self::$modules === false) {
            require_once APPLICATION_PATH . '/modules/system/SystemModule.php';

            self::$modules = ['system' => new SystemModule(true)];

            foreach (self::getModuleNames() as $module_name) {
                if ($module_name === SystemModule::NAME) {
                    continue;
                }

                $module_class = Inflector::camelize($module_name) . 'Module';
                require_once APPLICATION_PATH . "/modules/$module_name/$module_class.php";

                self::$modules[] = new $module_class(true, true);
            }
        }

        return self::$modules;
    }

    /**
     * Returns true if framework $name is loaded.
     *
     * @param  string $name
     * @return bool
     */
    public static function isFrameworkLoaded($name)
    {
        return isset(self::$loaded_frameworks_and_modules[$name]) && self::$loaded_frameworks_and_modules[$name] instanceof AngieFramework;
    }

    /**
     * Return module instance.
     *
     * @param  string            $name
     * @return AngieFramework
     * @throws InvalidParamError
     */
    public static function &getModule($name)
    {
        if (isset(self::$loaded_frameworks_and_modules[$name])) {
            return self::$loaded_frameworks_and_modules[$name];
        }

        throw new InvalidParamError('name', $name, "Module '$name' is not defined");
    }

    // ---------------------------------------------------
    //  File paths
    // ---------------------------------------------------

    /**
     * Find and include specific controller based on controller name.
     *
     * @param  string            $controller_name
     * @param  string            $module_name
     * @return string
     * @throws InvalidParamError
     */
    public static function useController($controller_name, $module_name = DEFAULT_MODULE)
    {
        if (isset(self::$loaded_frameworks_and_modules[$module_name])) {
            return self::$loaded_frameworks_and_modules[$module_name]->useController($controller_name);
        }

        throw new InvalidParamError('module_name', $module_name, "Module / framework '$module_name' not loaded");
    }

    /**
     * Use one or more models from a given module.
     *
     * @param  array             $model_names
     * @param  string            $module_name
     * @throws InvalidParamError
     */
    public static function useModel($model_names, $module_name = DEFAULT_MODULE)
    {
        if (isset(self::$loaded_frameworks_and_modules[$module_name])) {
            self::$loaded_frameworks_and_modules[$module_name]->useModel($model_names);
        } else {
            throw new InvalidParamError('module_name', $module_name, "Module / framework '$module_name' not loaded");
        }
    }

    /**
     * Use views.
     *
     * @param  array             $view_names
     * @param  string            $module_name
     * @throws InvalidParamError
     */
    public static function useView($view_names, $module_name)
    {
        if (empty($view_names)) {
            return;
        }

        if (isset(self::$loaded_frameworks_and_modules[$module_name])) {
            self::$loaded_frameworks_and_modules[$module_name]->useView($view_names);
        } else {
            throw new InvalidParamError('module_name', $module_name, "Module / framework '$module_name' not loaded");
        }
    }

    /**
     * Use helper file.
     *
     * @param  string            $helper_name
     * @param  string            $module_name
     * @param  string            $helper_type
     * @return string
     * @throws InvalidParamError
     */
    public static function useHelper($helper_name, $module_name = DEFAULT_MODULE, $helper_type = 'function')
    {
        if (isset(self::$loaded_frameworks_and_modules[$module_name])) {
            return self::$loaded_frameworks_and_modules[$module_name]->useHelper($helper_name, $helper_type);
        }

        throw new InvalidParamError('module_name', $module_name, "Module / framework '$module_name' not loaded");
    }

    // ---------------------------------------------------
    //  Paths
    // ---------------------------------------------------

    /**
     * Return URL for a given proxy with given parameters.
     *
     * @param  string            $proxy
     * @param  string            $module_name
     * @param  mixed             $params
     * @return string
     * @throws InvalidParamError
     */
    public static function getProxyUrl($proxy, $module_name = DEFAULT_MODULE, $params = null)
    {
        if (isset(self::$loaded_frameworks_and_modules[$module_name])) {
            return self::$loaded_frameworks_and_modules[$module_name]->getProxyUrl($proxy, $params);
        }

        throw new InvalidParamError('module_name', $module_name, "Module / framework '$module_name' not loaded");
    }

    /**
     * Return email template path.
     *
     * @param  string            $template
     * @param  string            $module_name
     * @return string
     * @throws InvalidParamError
     */
    public static function getEmailTemplatePath($template, $module_name = DEFAULT_MODULE)
    {
        if (isset(self::$loaded_frameworks_and_modules[$module_name])) {
            return self::$loaded_frameworks_and_modules[$module_name]->getEmailTemplatePath($template);
        }

        throw new InvalidParamError('module_name', $module_name, "Module / framework '$module_name' not loaded");
    }

    /**
     * Return handler file path based on event name.
     *
     * @param  string            $callback_name
     * @param  string            $module_name
     * @return string
     * @throws InvalidParamError
     */
    public static function getEventHandlerPath($callback_name, $module_name = DEFAULT_MODULE)
    {
        if (isset(self::$loaded_frameworks_and_modules[$module_name])) {
            return self::$loaded_frameworks_and_modules[$module_name]->getEventHandlerPath($callback_name);
        }

        throw new InvalidParamError('module_name', $module_name, "Module / framework '$module_name' not loaded");
    }

    // ---------------------------------------------------
    //  File management
    // ---------------------------------------------------

    /**
     * Return full file path based on file location.
     *
     * @param  string $location
     * @return string
     */
    public static function fileLocationToPath($location)
    {
        return UPLOAD_PATH . '/' . $location;
    }

    /**
     * Move or copy file to a permanent storage.
     *
     * Result is an array where first element is full path, and second is path relative to the upload folder
     *
     * @param  string        $path
     * @param  bool          $is_uploaded_file
     * @return array
     * @throws FileCopyError
     */
    public static function storeFile($path, $is_uploaded_file = false)
    {
        $target_path = self::prepareTargetPath();

        if ($is_uploaded_file ? move_uploaded_file($path, $target_path) : copy($path, $target_path)) {
            return [$target_path, substr($target_path, strlen(UPLOAD_PATH) + 1)];
        }

        throw new FileCopyError($path, $target_path);
    }

    /**
     * Remove stored file from disk.
     *
     * @param  string            $location
     * @param  string            $in
     * @throws InvalidParamError
     */
    public static function removeStoredFile($location, $in = UPLOAD_PATH)
    {
        if (empty($location)) {
            return; // Nothing to remove
        }

        if ($in !== UPLOAD_PATH && $in !== WORK_PATH) {
            throw new InvalidParamError('in', $in, '$in can be /upload or /work folder');
        }

        if (is_file($in . '/' . $location)) {
            @unlink($in . '/' . $location);
        }
    }

    /**
     * Prepare target path.
     *
     * @param  string               $in
     * @return string
     * @throws InvalidParamError
     * @throws DirectoryCreateError
     */
    private static function prepareTargetPath($in = UPLOAD_PATH)
    {
        if ($in !== UPLOAD_PATH && $in !== WORK_PATH) {
            throw new InvalidParamError('in', $in, '$in can be /upload or /work folder');
        }

        $target_path = $in . '/' . date('Y-m');

        if (!is_dir($target_path)) {
            $old_umask = umask(0000);
            $dir_created = mkdir($target_path, 0777);
            umask($old_umask);

            if (empty($dir_created)) {
                throw new DirectoryCreateError($target_path);
            }
        }

        do {
            $filename = $target_path . '/' . self::getAccountId() . '-' . make_string(40);
        } while (is_file($filename));

        return $filename;
    }

    /**
     * Return available file name in /uploads folder.
     *
     * @return string
     */
    public static function getAvailableUploadsFileName()
    {
        do {
            $filename = UPLOAD_PATH . '/' . self::getAccountId() . '-' . make_string(10) . '-' . make_string(10) . '-' . make_string(10) . '-' . make_string(10);
        } while (is_file($filename));

        return $filename;
    }

    /**
     * Prepare a directory under work path with proper permissions.
     *
     * @param  string $dir_path
     * @param  string $prefix
     * @return string
     */
    public static function getAvailableDirName($dir_path, $prefix)
    {
        if (!in_array($dir_path, [UPLOAD_PATH, WORK_PATH])) {
            throw new LogicException('This method is available only for upload and work directories');
        }

        if ($prefix) {
            $prefix = self::getAccountId() . "_{$prefix}_";
        } else {
            $prefix = self::getAccountId() . '_';
        }

        do {
            $target_dir_path = $dir_path . '/' . uniqid($prefix);
        } while (is_dir($target_dir_path));

        $old_umask = umask(0);
        mkdir($target_dir_path, 0777);
        umask($old_umask);

        return $target_dir_path;
    }

    /**
     * Return unique filename in work folder.
     *
     * @param  string $prefix
     * @param  string $extension
     * @param  bool   $random_string
     * @return string
     */
    public static function getAvailableWorkFileName($prefix = null, $extension = null, $random_string = true)
    {
        return self::getAvailableFileName(WORK_PATH, $prefix, $extension, $random_string);
    }

    /**
     * Get Available file name in $folder.
     *
     * @param  string $dir_path
     * @param  string $prefix
     * @param  string $extension
     * @param  bool   $random_string
     * @return string
     */
    public static function getAvailableFileName($dir_path, $prefix = null, $extension = null, $random_string = true)
    {
        if ($prefix) {
            $prefix = self::getAccountId() . "-{$prefix}-";
        } else {
            $prefix = self::getAccountId() . '-';
        }

        if ($extension) {
            $extension = ".$extension";
        }

        if ($random_string) {
            do {
                $filename = $dir_path . '/' . $prefix . make_string(10) . $extension;
            } while (is_file($filename));
        } else {
            $filename = trim($dir_path . '/' . $prefix, '-') . $extension;
        }

        return $filename;
    }

    // ---------------------------------------------------
    //  Wallpapers
    // ---------------------------------------------------

    /**
     * Get Wallpaper url.
     *
     * @param $name
     * @return string
     */
    public static function getWallpaperUrl($name)
    {
        return ROOT_URL . "/wallpapers/$name";
    }

    // ---------------------------------------------------
    //  Installation
    // ---------------------------------------------------

    /**
     * Returns true if this application is installed.
     *
     * @return bool
     */
    public static function isInstalled()
    {
        return defined('CONFIG_PATH') && is_file(CONFIG_PATH . '/config.php');
    }

    // ---------------------------------------------------
    //  Autoload
    // ---------------------------------------------------

    /**
     * Array of registered classes that autoloader uses.
     *
     * @var array
     */
    private static $autoload_classes = [];

    /**
     * Automatically load requested class.
     *
     * @param  string        $class
     * @throws AutoloadError
     */
    public static function autoload($class)
    {
        if ($class === 'CURLFile') {
            return;
        }

        $path = array_var(self::$autoload_classes, $class);

        if ($path && is_file($path)) {
            require_once $path;
        } else {
            if (stripos($class, 'smarty') !== false) {
                return; // Ignore Smarty classes
            }

            if ($class === 'Simple') {
                return; // Ignore Elastica factory check for Simple strategy @TODO patch when covered
            }

            throw new AutoloadError($class);
        }
    }

    /**
     * Register class to autoload array.
     *
     * $class can be an array of classes, where index is class name value is
     * path to the file where class is defined
     *
     * @param  array $classes_and_paths
     * @throws Error
     */
    public static function setForAutoload(array $classes_and_paths)
    {
        foreach ($classes_and_paths as $class => $path) {
            if (!empty(self::$autoload_classes[$class])) {
                throw new Error("Class '$class' already set for autoload (currently points to '" . self::$autoload_classes[$class] . "')");
            }

            self::$autoload_classes[$class] = $path;
        }
    }

    /**
     * Return used memory from moment script was loaded until now.
     *
     * @return float
     */
    public static function getMemoryUsage()
    {
        return memory_get_peak_usage(true);
    }

    /**
     * Return time spent from moment script was loaded until now.
     *
     * @return float
     */
    public static function getExecutionTime()
    {
        if (!defined('ANGIE_SCRIPT_TIME')) {
            throw new RuntimeException('Reference timestamp constant (ANGIE_SCRIPT_TIME) not found');
        }

        return round(microtime(true) - ANGIE_SCRIPT_TIME, 5);
    }

    /**
     * Called on application shutdown.
     */
    public static function shutdown()
    {
        try {
            if (self::$global_job_queue_connection instanceof mysqli) {
                self::$global_job_queue_connection->close();
            }

            if (DB::hasConnection('default') && DB::getConnection('default')->isConnected()) {
                DB::getConnection('default')->disconnect(); // Lets disconnect and kill a transaction if we have something open
            }

            if (AngieApplication::isOnDemand()) {
                aCID::getInstance(false)->disconnect();
            }

            self::log()->requestSummary(self::getExecutionTime(), self::getMemoryUsage(), DB::getQueryCount(), DB::getAllQueriesExecTime());

            if (!empty(self::log()->getBuffer())) {
                self::log()->flushBuffer(true);
            }
        } catch (Exception $e) {
            if (!self::isInProduction()) {
                throw $e;
            }
            trigger_error('Error detected on shutdown: ' . $e->getMessage());
        }
    }

    /**
     * TODO -- remove this workaround for AngieDelateCache.
     */
    public static function initializeOnDemandModel()
    {
        require_once APPLICATION_PATH . '/modules/on_demand/models/OnDemand.class.php';
    }
}
