<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

/**
 * Check environment prior to installation / upgrade.
 *
 * @package angie.library.application
 */
final class AngieApplicationEnvironmentChecker
{
    /**
     * @var string
     */
    private $min_php_version = '5.6.0';

    /**
     * @var array
     */
    private $log = [];

    /**
     * @var bool
     */
    private $all_ok = true;

    /**
     * @var callable
     */
    private $on_pass;

    /**
     * @var callable
     */
    private $on_fail;

    /**
     * @param  callable|null $on_pass
     * @param  callable|null $on_fail
     * @return bool
     */
    public function check(callable $on_pass = null, callable $on_fail = null)
    {
        $this->log = [];
        $this->all_ok = true;
        $this->on_pass = $on_pass;
        $this->on_fail = $on_fail;

        if (version_compare(PHP_VERSION, $this->min_php_version, '>=')) {
            $this->pass('Your PHP is ' . PHP_VERSION);
        } else {
            $this->fail('PHP version that is required to run the system is ' . $this->min_php_version . '. You have ' . PHP_VERSION);
        }

        if (extension_loaded('phar')) {
            $this->pass('Phar extension is available');
        } else {
            $this->fail('Required Phar PHP extension was not found. Please install it before continuing');
        }

        if (extension_loaded('zlib')) {
            $this->pass('zlib extension is available');
        } else {
            $this->fail('Required zlib PHP extension was not found. Please install it before continuing');
        }

        $memory_limit = php_config_value_to_bytes(ini_get('memory_limit'));

        if ($memory_limit == -1 || $memory_limit >= 67108864) {
            $formatted_memory_limit = $memory_limit == -1 ? 'unlimited' : format_file_size($memory_limit);
            $this->pass('Your memory limit is ' . $formatted_memory_limit);
        } else {
            $this->fail('Your memory is too low to complete the upgrade. Minimal value is 64MB, and you have it set to ' . format_file_size($memory_limit));
        }

        if (folder_is_writable(ROOT)) {
            $this->pass('/' . basename(ROOT) . ' directory is writable');
        } else {
            $this->fail('/' . basename(ROOT) . ' directory is not writable. Make it writable to continue');
        }

        if (folder_is_writable(ASSETS_PATH)) {
            $this->pass('/public/assets is writable');
        } else {
            $this->fail('/public/assets is not writable. Make it writable to continue');
        }

        if (file_is_writable(CONFIG_PATH . '/version.php')) {
            $this->pass('/config/version.php is writable');
        } else {
            $this->fail('/config/version.php is not writable. Make it writable to continue');
        }

        return $this->all_ok;
    }

    /**
     * @param string $message
     */
    public function pass($message)
    {
        $this->log[] = ['ok' => true, 'message' => $message];

        if ($this->on_pass && is_callable($this->on_pass)) {
            call_user_func($this->on_pass, $message);
        }
    }

    /**
     * @param string $message
     */
    public function fail($message)
    {
        $this->all_ok = false;
        $this->log[] = ['ok' => false, 'message' => $message];

        if ($this->on_fail && is_callable($this->on_fail)) {
            call_user_func($this->on_fail, $message);
        }
    }
}
