<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

/**
 * Basic calendar feed implementation.
 *
 * @package angie.frameworks.calendars
 * @subpackage models
 */
trait ICalendarFeedImplementation
{
    /**
     * Export calendar to iCalendar file.
     *
     * @param  User            $user
     * @return string
     * @throws FileCreateError
     */
    public function exportCalendarToFile(User $user)
    {
        $filename = $this->getCalendarExportFilename($user);

        if (!is_file($filename)) {
            if ($handle = fopen($filename, 'w')) {
                try {
                    $vcalendar = $this->getVCalendarObject();

                    $summary_prefix = $this->getCalendarElementSummaryPrefix();
                    $summary_sufix = $this->getCalendarElementSummarySufix();

                    foreach ($this->getCalendarFeedElements($user) as $calendar_feed_element) {
                        $calendar_feed_element->exportToCalendar($vcalendar, $user, $summary_prefix, $summary_sufix);
                    }

                    fwrite($handle, $vcalendar->serialize());
                } finally {
                    fclose($handle);
                }
            } else {
                throw new FileCreateError($filename);
            }
        }

        return $filename;
    }

    /**
     * {@inheritdoc}
     */
    public function getCalendarElementSummaryPrefix()
    {
        return '';
    }

    /**
     * {@inheritdoc}
     */
    public function getCalendarElementSummarySufix()
    {
        return '';
    }

    /**
     * Create and return new VCalendar object instance.
     *
     * @return \Sabre\VObject\Component\VCalendar
     */
    protected function getVCalendarObject()
    {
        return new \Sabre\VObject\Component\VCalendar();
    }

    /**
     * Return calendar elements that $user has access to.
     *
     * @param  IUser                  $user
     * @return ICalendarFeedElement[]
     */
    abstract protected function getCalendarFeedElements(IUser $user);

    /**
     * Return proposed calendar file name.
     *
     * For objects that implemented IUpdatedOn behavior, system will return:
     *
     * type-#CALENDAR_ID#-for-#USER_ID#-#UPDATED_ON#.ics
     *
     * If object does not implement IUpdatedOn, system will return:
     *
     * type-#CALENDAR_ID#-for-#USER_ID#.ics
     *
     * @param  User   $user
     * @return string
     */
    protected function getCalendarExportFilename(User $user)
    {
        $bits = [
            AngieApplication::getAccountId(),
            $this->getModelName(false, true),
            $this->getId(),
        ];

        if ($this instanceof IUpdatedOn) {
            $bits[] = $this->getUpdatedOn()->getTimestamp();
        }

        $bits[] = '-for-';
        $bits[] = $user->getId();
        $bits[] = $user->getUpdatedOn()->getTimestamp();

        return WORK_PATH . '/' . implode('-', $bits) . '.ics';
    }

    /**
     * Return object ID.
     *
     * @return int
     */
    abstract public function getId();

    /**
     * Return name of this model.
     *
     * @param  bool   $underscore
     * @param  bool   $singular
     * @return string
     */
    abstract public function getModelName($underscore = false, $singular = false);
}
