<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use Angie\Globalization;

/**
 * Cron integration.
 *
 * @package angie.frameworks.environment
 * @subpackage models
 */
class CronIntegration extends Integration
{
    const HOURLY_MAINTENANCE = 'hourly_maintenance';
    const DAILY_MAINTENANCE = 'daily_maintenance';
    const MORNING_MAIL = 'morning_mail';

    /**
     * Returns true if this integration is singleton (can be only one integration of this type in the system).
     *
     * @return bool
     */
    public function isSingleton()
    {
        return true;
    }

    /**
     * Returns true if this integration is in use.
     *
     * @return bool
     */
    public function isInUse(User $user = null)
    {
        return true;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return 'Cron Jobs';
    }

    /**
     * Return integration short name.
     *
     * @return string
     */
    public function getShortName()
    {
        return 'cron';
    }

    /**
     * Return short integration description.
     *
     * @return string
     */
    public function getDescription()
    {
        return lang('Use Cron jobs to perform various system tasks');
    }

    /**
     * Return true if this integration is available for on-demand packages.
     *
     * @return bool
     */
    public function isAvailableForOnDemand()
    {
        return false; // Use pre-configured mailing and don't let settings be changed via API
    }

    /**
     * @return array
     */
    public function jsonSerialize()
    {
        $php_executable_path = $this->getExecutableFromPaths();

        $frequently_command = $php_executable_path . ' ' . escapeshellarg(ENVIRONMENT_PATH . '/tasks/cron_jobs/run_every_minute.php');
        $hourly_command = $php_executable_path . ' ' . escapeshellarg(ENVIRONMENT_PATH . '/tasks/cron_jobs/run_every_hour.php');
        $check_imap_command = $php_executable_path . ' ' . escapeshellarg(ENVIRONMENT_PATH . '/tasks/cron_jobs/check_imap_every_3_minutes.php');

        if (DIRECTORY_SEPARATOR == '\\') {
            $platform = 'windows';

            $setup_note = 'IIS';
            $setup_frequently = 'schtasks /create /ru "IIS" /sc minute /tn "Active Collab Frequently Job" /tr "' . $frequently_command . '"';
            $setup_hourly = 'schtasks /create /ru "IIS" /sc hourly /st 12:00:00 /tn "Active Collab Hourly Job" /tr "' . $hourly_command . '"';
            $setup_check_imap = 'schtasks /create /ru "IIS" /sc minute /mo 3 /tn "Active Collab Inbound Email Job" /tr "' . $check_imap_command . '"';
        } else {
            $platform = 'unix';

            $setup_note = '';
            $setup_frequently = '(crontab -l ; echo "* * * * * ' . $frequently_command . '") | sort - | uniq - | crontab -';
            $setup_hourly = '(crontab -l ; echo "0 * * * * ' . $hourly_command . '") | sort - | uniq - | crontab -';
            $setup_check_imap = '(crontab -l ; echo "*/3 * * * * ' . $check_imap_command . '") | sort - | uniq - | crontab -';
        }

        return array_merge(parent::jsonSerialize(), [
            'maintenance_at' => 4,
            'morning_mail_at' => 7,
            'instructions' => [
                'platform' => $platform,
                'frequently' => $frequently_command,
                'hourly' => $hourly_command,
                'imap_check' => $check_imap_command,
                'setup_note' => $setup_note,
                'setup_frequently' => $setup_frequently,
                'setup_hourly' => $setup_hourly,
                'setup_check_imap' => $setup_check_imap,
                'frequently_last_run' => AngieApplication::memories()->get('frequently_last_run', null, false),
                'hourly_last_run' => AngieApplication::memories()->get('hourly_last_run', null, false),
                'check_imap_last_run' => AngieApplication::memories()->get('check_imap_last_run', null, false),
            ],
        ]);
    }

    /**
     * Return PATH environment variable and try to get PHP binary path from.
     *
     * @return string
     */
    public function getExecutableFromPaths()
    {
        if (DISCOVER_PHP_CLI) {
            $exacutable_names = ['php', 'php-cli', 'php56', 'php55', 'php54'];

            foreach (explode(PATH_SEPARATOR, getenv('PATH')) as $path) {
                foreach ($exacutable_names as $exacutable_name) {
                    if (strstr($path, "{$exacutable_name}.exe") && isset($_SERVER['WINDIR']) && file_exists($path) && is_file($path) && $this->checkPhpVersion($path)) {
                        return $path;
                    } else {
                        $php_executable = $path . DIRECTORY_SEPARATOR . $exacutable_name;

                        if (isset($_SERVER['WINDIR'])) {
                            $php_executable .= '.exe';
                        }

                        if (file_exists($php_executable) && is_file($php_executable) && $this->checkPhpVersion($php_executable)) {
                            return $php_executable;
                        }
                    }
                }
            }
        }

        return DIRECTORY_SEPARATOR == '\\' ? 'C:\\path\\to\\php.exe' : '/path/to/your/php';
    }

    /**
     * Try to run php -v for the given binary and check the output.
     *
     * @param  string $executable_path
     * @return bool
     */
    private function checkPhpVersion($executable_path)
    {
        $output = [];
        $exit_code = 0;

        exec("$executable_path -v", $output, $exit_code);

        if ($exit_code === 0 && count($output)) {
            $bits = explode(' ', $output[0]);

            return count($bits) >= 2 && $bits[0] == 'PHP' && version_compare($bits[1], '5.4') >= 0;
        }

        return false;
    }

    /**
     * Returns true if cron jobs are configured properly and are being fired.
     *
     * @param  array $error_messages
     * @return bool
     */
    public function isOk(array &$error_messages = null)
    {
        $current_timestamp = time();

        if ((int) AngieApplication::memories()->get('frequently_last_run', null, false) < $current_timestamp - 300) {
            if ($error_messages === null) {
                return false;
            } else {
                $error_messages[] = lang('Frequently Cron job did not run in the past 5 minutes');
            }
        }

        if ((int) AngieApplication::memories()->get('check_imap_last_run', null, false) < $current_timestamp - 300) {
            if ($error_messages === null) {
                return false;
            } else {
                $error_messages[] = lang('Inbound email Cron job did not run in the past 5 minutes');
            }
        }

        if ((int) AngieApplication::memories()->get('hourly_last_run', null, false) < $current_timestamp - 3900) {
            if ($error_messages === null) {
                return false;
            } else {
                $error_messages[] = lang('Hourly Cron job did not run in the past hour');
            }
        }

        return empty($error_messages);
    }

    // ---------------------------------------------------
    //  Maintenance and Morining Mail
    // ---------------------------------------------------

    /**
     * Run every hour.
     *
     * @param int      $timestamp
     * @param callable $output
     */
    public function runEveryHour($timestamp, callable $output)
    {
        if (DB::executeFirstCell("SELECT COUNT(`id`) AS 'row_count' FROM `memories` WHERE `key` = 'hourly_last_run'")) {
            DB::execute("UPDATE `memories` SET `value` = ?, `updated_on` = ? WHERE `key` = 'hourly_last_run'", serialize($timestamp), date('Y-m-d H:i:s'));
        } else {
            DB::execute("INSERT INTO `memories` (`key`, `value`, `updated_on`) VALUES ('hourly_last_run', ?, ?)", serialize($timestamp), date('Y-m-d H:i:s'));
        }

        switch ($this->whichEventShouldBeDoneAt($timestamp)) {
            case self::DAILY_MAINTENANCE:
                call_user_func($output, 'Performing daily maintenance');
                $this->dailyMaintenance($timestamp);
                break;
            case self::MORNING_MAIL:
                call_user_func($output, 'Sending morning mail');
                $this->sendMorningMail($timestamp);
                break;
        }

        call_user_func($output, 'Performing hourly maintenance');
        $this->hourlyMaintenance();
    }

    /**
     * Return which event shiuld be performed at the given timestamp.
     *
     * @param  int    $timestamp
     * @return string
     */
    public function whichEventShouldBeDoneAt($timestamp)
    {
        if ($this->shouldDoDailyMaintenance($timestamp)) {
            return self::DAILY_MAINTENANCE;
        } else {
            if ($this->shouldSendMorningMail($timestamp)) {
                return self::MORNING_MAIL;
            } else {
                return self::HOURLY_MAINTENANCE;
            }
        }
    }

    /**
     * Return true if maintenance should be performed for the given timestamp.
     *
     * @param  int  $timestamp
     * @return bool
     */
    public function shouldDoDailyMaintenance($timestamp)
    {
        return $this->shouldDoEvent(self::DAILY_MAINTENANCE, 4, $timestamp);
    }

    /**
     * Return true if maintenance should be performed for the given timestamp.
     *
     * @param  string $event
     * @param  int    $it_should_be_done_at
     * @param  int    $timestamp
     * @return bool
     */
    private function shouldDoEvent($event, $it_should_be_done_at, $timestamp)
    {
        if ($this->isEventDone($event, $timestamp)) {
            return false;
        }

        $hour = date('H', $this->getLocalTimestamp($timestamp));

        return $hour >= $it_should_be_done_at && $hour < 12;
    }

    // ---------------------------------------------------
    //  Event Logs Utils
    // ---------------------------------------------------

    /**
     * Return true if maintenance has already been performed for the given date.
     *
     * @param  string $event
     * @param  int    $timestamp
     * @return bool
     */
    private function isEventDone($event, $timestamp)
    {
        $log = AngieApplication::memories()->get("{$event}_log");

        return $log && isset($log[date('Y-m-d', $this->getLocalTimestamp($timestamp))]);
    }

    /**
     * Correct timestamp with system's GMT offset.
     *
     * @param  int $timestamp
     * @return int
     */
    private function getLocalTimestamp($timestamp)
    {
        return $timestamp + Globalization::getGmtOffset();
    }

    /**
     * Return true if maintenance should be performed for the given timestamp.
     *
     * @param  int  $timestamp
     * @return bool
     */
    public function shouldSendMorningMail($timestamp)
    {
        return $this->shouldDoEvent(self::MORNING_MAIL, 7, $timestamp);
    }

    /**
     * Perform maintenance tasks.
     *
     * @param int $timestamp
     */
    public function dailyMaintenance($timestamp)
    {
        $this->setDailyMaintenanceDone($timestamp, 'Trying to perform maintenance');
        Angie\Events::trigger('on_daily_maintenance');
        $this->setDailyMaintenanceDone($timestamp);

        $time_to_perform = round(microtime(true) - ANGIE_SCRIPT_TIME, 5);

        if (AngieApplication::isEdgeChannel() && AngieApplication::getAccountId() === 1) {
            /** @var InfoNotification $info_notification */
            $info_notification = AngieApplication::notifications()->notifyAbout(EnvironmentFramework::INJECT_INTO . '/info');
            $info_notification->setCustomSubject('Maintenance Performed');
            $info_notification->setCustomMessage('Maintenance for ' . ROOT_URL . " has been successfully performed. Time taken: {$time_to_perform}s");
            $info_notification->sendToUsers(new Owner(1), true);
        }
    }

    /**
     * Set a daily maintenance as done for a given day.
     *
     * @param int         $timestamp
     * @param string|null $error_message
     */
    public function setDailyMaintenanceDone($timestamp, $error_message = null)
    {
        $this->setEventDone(self::DAILY_MAINTENANCE, $timestamp, $error_message);
    }

    // ---------------------------------------------------
    //  Maintenance
    // ---------------------------------------------------

    /**
     * Set a maintenance or morning mail as done for a given day.
     *
     * @param string      $event
     * @param int         $timestamp
     * @param string|null $error_message
     */
    private function setEventDone($event, $timestamp, $error_message = null)
    {
        $log = AngieApplication::memories()->get("{$event}_log");

        if (empty($log)) {
            $log = [];
        }

        $log[date('Y-m-d', $this->getLocalTimestamp($timestamp))] = $error_message ? ['ok' => false, 'error_message' => $error_message] : ['ok' => true];

        AngieApplication::memories()->set("{$event}_log", $log);
    }

    /**
     * Send morning mail.
     *
     * @param int $timestamp
     */
    public function sendMorningMail($timestamp)
    {
        $this->setMorningMailDone($timestamp, 'Trying to send morning mail');
        Angie\Events::trigger('on_morning_mail');
        $this->setMorningMailDone($timestamp);

        $time_to_send = round(microtime(true) - ANGIE_SCRIPT_TIME, 5);

        if (AngieApplication::isEdgeChannel() && AngieApplication::getAccountId() === 1) {
            /** @var InfoNotification $info_notification */
            $info_notification = AngieApplication::notifications()->notifyAbout(EnvironmentFramework::INJECT_INTO . '/info');
            $info_notification->setCustomSubject('Morning Mail Sent');
            $info_notification->setCustomMessage('Morning mail for ' . ROOT_URL . " has successfuly been sent. Time taken: {$time_to_send}s");
            $info_notification->sendToUsers(new Owner(1), true);
        }
    }

    /**
     * Set a maintenance as done for a given day.
     *
     * @param int         $timestamp
     * @param string|null $error_message
     */
    public function setMorningMailDone($timestamp, $error_message = null)
    {
        $this->setEventDone(self::MORNING_MAIL, $timestamp, $error_message);
    }

    // ---------------------------------------------------
    //  Morning Mail
    // ---------------------------------------------------

    /**
     * Perform hourly maintenance.
     */
    public function hourlyMaintenance()
    {
        Angie\Events::trigger('on_hourly_maintenance');
    }

    /**
     * Return true if maintenance has already been performed for the given date.
     *
     * @param  int  $timestamp
     * @return bool
     */
    public function isDailyMaintenanceDone($timestamp)
    {
        return $this->isEventDone(self::DAILY_MAINTENANCE, $timestamp);
    }

    /**
     * Return true if maintenance has already been performed for the given date.
     *
     * @param  int  $timestamp
     * @return bool
     */
    public function isMorningMailSent($timestamp)
    {
        return $this->isEventDone(self::MORNING_MAIL, $timestamp);
    }
}
