<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use ActiveCollab\ActiveCollabJobs\Jobs\Search\TestConnection;
use Angie\Search;

/**
 * Search integration (ElasticSearch powered).
 *
 * @package angie.frameworks.environment
 * @subpackage models
 */
class SearchIntegration extends Integration
{
    const JOBS_QUEUE_CHANNEL = 'search';

    /**
     * Returns true if this integration is singleton (can be only one integration of this type in the system).
     *
     * @return bool
     */
    public function isSingleton()
    {
        return true;
    }

    /**
     * Returns true if this integration is in use.
     *
     * @return bool
     */
    public function isInUse(User $user = null)
    {
        return true;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return 'Search';
    }

    /**
     * Return integration short name.
     *
     * @return string
     */
    public function getShortName()
    {
        return 'search';
    }

    /**
     * Return short integration description.
     *
     * @return string
     */
    public function getDescription()
    {
        return lang('Use ElasticSearch to index and search your data');
    }

    /**
     * Return true if this integration is available for on-demand packages.
     *
     * @return bool
     */
    public function isAvailableForOnDemand()
    {
        return false; // Use pre-configured mailing and don't let settings be changed via API
    }

    /**
     * Test connection to ElasticSearch server.
     *
     * @param  string    $host
     * @param  int       $port
     * @param  bool|true $execute_now
     * @return mixed
     */
    public function testConnection($host, $port, $execute_now = true)
    {
        $job = new TestConnection([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
        ]);

        if ($execute_now) {
            return AngieApplication::jobs()->execute($job);
        } else {
            return AngieApplication::jobs()->dispatch($job, self::JOBS_QUEUE_CHANNEL); // Used just for job creation testing. Connection testing in the background is kind of pointless
        }
    }

    /**
     * @return string
     */
    public function getHost()
    {
        return $this->getAdditionalProperty('host');
    }

    /**
     * @param  string $value
     * @return string
     */
    public function setHost($value)
    {
        return $this->setAdditionalProperty('host', trim($value));
    }

    /**
     * @return int
     */
    public function getPort()
    {
        return $this->getAdditionalProperty('port', 9200);
    }

    /**
     * @param  int $value
     * @return int
     */
    public function setPort($value)
    {
        return $this->setAdditionalProperty('port', (int) $value);
    }

    /**
     * @return array
     */
    public function jsonSerialize()
    {
        return array_merge(parent::jsonSerialize(), [
            'host' => $this->getHost(),
            'port' => $this->getPort(),
        ]);
    }

    /**
     * Returns true if search is configured properly.
     *
     * @param  array $error_messages
     * @return bool
     */
    public function isOk(array &$error_messages = null)
    {
        if (!$this->getHost()) {
            if ($error_messages === null) {
                return false;
            } else {
                $error_messages[] = lang('Search server is not configured');
            }
        }

        return empty($error_messages);
    }

    // ---------------------------------------------------
    //  System
    // ---------------------------------------------------

    /**
     * Save to database and reset Search index if we are reconnecting.
     */
    public function save()
    {
        if ($this->isModifiedField('raw_additional_properties')) {
            $additional_properties_updated = true;

            $old_properties = $this->getOldFieldValue('raw_additional_properties');
            $old_properties = $old_properties ? unserialize($old_properties) : [];

            $old_host = isset($old_properties['host']) && $old_properties['host'] ? $old_properties['host'] : '';
            $old_port = isset($old_properties['port']) && $old_properties['port'] ? $old_properties['port'] : 9200;
        } else {
            $additional_properties_updated = false;
            $old_host = null;
            $old_port = null;
        }

        parent::save();

        $new_host = $this->getHost();

        if ($additional_properties_updated && $new_host && ($new_host != $old_host || $this->getPort() != $old_port)) {
            Search::reset();
        }
    }
}
