<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

/**
 * Envrionment framework model definition.
 *
 * @package angie.frameworks.environment
 * @subpackage resources
 */
class EnvironmentFrameworkModel extends AngieFrameworkModel
{
    /**
     * Construct environment framework model definition.
     *
     * @param EnvironmentFramework $parent
     */
    public function __construct(EnvironmentFramework $parent)
    {
        parent::__construct($parent);

        $this->addTableFromFile('config_options');
        $this->addTableFromFile('config_option_values');

        $this->addModelFromFile('uploaded_files')->setTypeFromField('type');
        $this->addTableFromFile('executed_model_migrations');
        $this->addTableFromFile('job_batches');
        $this->addTableFromFile('jobs_queue');
        $this->addTableFromFile('jobs_queue_failed');
        $this->addTableFromFile('memories');

        $this->addModelFromFile('system_notifications')->setTypeFromField('type')->setObjectIsAbstract(true)->setOrderBy('created_on DESC, id DESC');

        $this->addModel(DB::createTable('access_logs')->addColumns([
            DBIdColumn::create()->setSize(DBColumn::BIG),
            DBParentColumn::create(),
            DBUserColumn::create('accessed_by', true),
            DBDateTimeColumn::create('accessed_on'),
            DBStringColumn::create('ip_address', 50),
            DBBoolColumn::create('is_download', false),
        ])->addIndices([
            DBIndex::create('accessed_on'),
        ]));

        $this->addTable(DB::createTable('routing_cache')->addColumns([
            DBIdColumn::create(),
            DBStringColumn::create('path_info', DBStringColumn::MAX_LENGTH),
            DBStringColumn::create('name', DBStringColumn::MAX_LENGTH),
            DBTextColumn::create('content'),
            DBDateTimeColumn::create('last_accessed_on'),
        ])->addIndices([
            DBIndex::create('path_info', DBIndex::UNIQUE),
        ]));

        $this->addModel(DB::createTable('currencies')->addColumns([
            DBIdColumn::create(),
            DBNameColumn::create(50, true),
            DBStringColumn::create('code', 3),
            DBStringColumn::create('symbol', 5),
            DBStringColumn::create('symbol_native', 5),
            DBIntegerColumn::create('decimal_spaces', 1, 2)->setUnsigned(true),
            DBDecimalColumn::create('decimal_rounding', 4, 3, '0.000')->setUnsigned(true),
            DBBoolColumn::create('is_default', false),
            DBUpdatedOnColumn::create(),
        ])->addIndices([
            DBIndex::create('code', DBIndex::UNIQUE),
        ]))->setOrderBy('name')
            ->addModelTrait(null, 'IResetInitialSettingsTimestamp');

        $this->addModel(DB::createTable('data_filters')->addColumns([
            DBIdColumn::create(),
            DBTypeColumn::create('DataFilter', 150),
            DBNameColumn::create(),
            DBAdditionalPropertiesColumn::create(),
            DBCreatedOnByColumn::create(),
            DBUpdatedOnColumn::create(),
            DBBoolColumn::create('is_private', false),
        ])->addIndices([
            DBIndex::create('name'),
        ]))->setTypeFromField('type')->setOrderBy('name')->addModelTrait('\Angie\Reports\Report', '\Angie\Reports\Report\Implementation');

        $this->addModel(DB::createTable('integrations')->addColumns([
            DBIdColumn::create(),
            DBTypeColumn::create(),
            DBAdditionalPropertiesColumn::create(),
            DBCreatedOnByColumn::create(),
        ]))->setTypeFromField('type')->setObjectIsAbstract(true);

        $this->addModel(DB::createTable('webhooks')->addColumns([
            DBIdColumn::create(),
            DBFkColumn::create('integration_id'),
            DBNameColumn::create(100),
            DBStringColumn::create('url'),
            DBBoolColumn::create('is_enabled'),
            DBStringColumn::create('secret'),
            DBTextColumn::create('filter_event_types'),
            DBTextColumn::create('filter_projects'),
            DBCreatedOnByColumn::create(true, true),
        ]));

        $this->addModel(DB::createTable('day_offs')->addColumns([
            DBIdColumn::create(),
            DBNameColumn::create(100),
            DBDateColumn::create('start_date'),
            DBDateColumn::create('end_date'),
            DBBoolColumn::create('repeat_yearly', false),
        ])->addIndices([
            DBIndex::create('day_off_name', DBIndex::UNIQUE, ['name', 'start_date', 'end_date']),
        ]))->setOrderBy('start_date');

        $this->addTable(DB::createTable('favorites')->addColumns([
            DBParentColumn::create(false),
            DBIntegerColumn::create('user_id', 10)->setUnsigned(true),
        ])->addIndices([
            DBIndex::create('favorite_object', DBIndex::UNIQUE, ['parent_type', 'parent_id', 'user_id']),
            DBIndex::create('user_id'),
        ]));

        $this->addModel(DB::createTable('languages')->addColumns([
            DBIdColumn::create(),
            DBNameColumn::create(50, true),
            DBStringColumn::create('locale', 30, ''),
            DBStringColumn::create('decimal_separator', 1, '.'),
            DBStringColumn::create('thousands_separator', 1, ','),
            DBBoolColumn::create('is_rtl'),
            DBBoolColumn::create('is_community_translation'),
            DBBoolColumn::create('is_default'),
            DBUpdatedOnColumn::create(),
        ])->addIndices([
            DBIndex::create('locale', DBIndex::UNIQUE),
        ]))->setOrderBy('name')
            ->addModelTrait(null, 'IResetInitialSettingsTimestamp');

        if (is_file(__DIR__ . '/table.test_data_objects.php')) {
            $this->addModelFromFile('test_data_objects')->setOrderBy('name')->setTypeFromField('type');
        }
    }

    /**
     * Load initial framework data.
     */
    public function loadInitialData()
    {
        $this->addConfigOption('maintenance_message');

        $this->addConfigOption('help_improve_application', true);
        $this->addConfigOption('require_index_rebuild', false);

        $this->addConfigOption('identity_name', 'Application');
        $this->addConfigOption('identity_logo');

        $this->addConfigOption('whitelisted_tags', [
            'environment' => [
                'a' => ['href', 'title', 'class', 'object-id', 'object-class'],
                'div' => ['class', 'placeholder-type', 'placeholder-object-id', 'placeholder-extra'],
                'img' => ['src', 'alt', 'title', 'class', 'image-type', 'object-id'],
                'p' => ['class', 'style'],
                'strike' => ['class', 'style'],
                'span' => ['class', 'data-redactor-inlinemethods', 'data-redactor', 'object-id'],
                'pre' => ['data-syntax'],
                'blockquote' => null,
                'br' => null,
                'b' => null,
                'strong' => null,
                'i' => null,
                'em' => null,
                'u' => null,
            ],
        ]);

        // wallpaper
        $this->addConfigOption('homepage');
        $this->addConfigOption('wallpaper', 'wallpaper.jpg');

        // Globalizatioj
        $this->addConfigOption('time_first_week_day', 0);
        $this->addConfigOption('time_timezone_autodetect', true);
        $this->addConfigOption('time_timezone', 'UTC');
        $this->addConfigOption('time_workdays', [1, 2, 3, 4, 5]);

        $this->addConfigOption('format_date', '%b %e. %Y');
        $this->addConfigOption('format_time', '%I:%M %p');

        $this->addConfigOption('initial_settings_timestamp', time());

        $this->addConfigOption('show_visual_editor_toolbar', false);

        // ---------------------------------------------------
        //  Languages, currencies
        // ---------------------------------------------------

        $this->loadTableData('languages', [
            ['name' => 'English', 'locale' => 'en_US.UTF-8', 'decimal_separator' => '.', 'thousands_separator' => ',', 'is_default' => true, 'updated_on' => date(DATETIME_MYSQL)],
        ]);

        $currencies = [];

        foreach (json_decode(file_get_contents(ANGIE_PATH . '/frameworks/environment/resources/Common-Currency.json'), true) as $currency_code => $currency_details) {
            $currencies[] = DB::prepare('(?, ?, ?, ?, ?, ?)', $currency_details['name'], $currency_code, $currency_details['symbol'], $currency_details['symbol_native'], $currency_details['decimal_digits'], $currency_details['rounding']);
        }

        DB::execute('INSERT INTO currencies (name, code, symbol, symbol_native, decimal_spaces, decimal_rounding) VALUES ' . implode(', ', $currencies));

        DB::execute('UPDATE currencies SET updated_on = UTC_TIMESTAMP()');
        DB::execute('UPDATE currencies SET is_default = ? WHERE code = ?', true, 'USD');

        parent::loadInitialData();
    }
}
