<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace Angie\Search\Adapter\ElasticSearch;

use Angie\Events;
use Angie\Search;
use Angie\Search\Criterion;
use DateTimeValue;
use Elastica\Filter\AbstractFilter;
use Elastica\Filter\BoolAnd;
use Elastica\Filter\BoolFilter;
use Elastica\Filter\BoolOr;
use Elastica\Filter\Range;
use Elastica\Filter\Term;
use Elastica\Filter\Terms;
use Elastica\Query;
use Elastica\Query\Filtered as FilteredQuery;
use Elastica\Query\FunctionScore;
use Elastica\Query\Prefix;
use Elastica\Query\QueryString;
use Elastica\Suggest;
use Elastica\Util;
use User;

trait PrepareQueries
{
    /**
     * Suggest the terms for the $search_for.
     *
     * @param  string $search_for
     * @param  User   $user
     * @return array
     */
    public function suggest($search_for, User $user)
    {
        if ($trigger = $this->shouldAutocomplete($search_for)) {
            return $this->autocomplete($trigger, $search_for, $user);
        } else {
            return $this->suggestName($search_for, $user);
        }
    }

    /**
     * Prepare search query.
     *
     * @param  string      $search_for
     * @param  User        $user
     * @param  Criterion[] $criterions
     * @param  int         $page
     * @return Query
     */
    private function prepareQuery($search_for, User $user, $criterions = null, $page = 1)
    {
        $query = new Query();
        $query->setSize(100);

        if ($page > 1) {
            $query->setFrom(($page - 1) * 100);
        }

        $query_string = new QueryString(Util::escapeTerm($search_for));
        $query_string->setDefaultOperator('AND');

        if (($filter = $this->prepareQueryFilter($user, $criterions)) instanceof AbstractFilter) {
            $filted_query = new FilteredQuery();

            $filted_query->setFilter($filter);
            $filted_query->setQuery($query_string);

            $query->setQuery($filted_query);
        } else {
            $query->setQuery($query_string);
        }

        $query->setHighlight([
            'pre_tags' => ['<em class="highlight">'],
            'post_tags' => ['</em>'],
            'fields' => [
                'name' => ['fragment_size' => 255, 'number_of_fragments' => 1],
                'body' => ['fragment_size' => 80, 'number_of_fragments' => 5],
            ],
        ]);

        return $query;
    }

    /**
     * Prepare suggest name query.
     *
     * @param  string        $search_for
     * @param  User          $user
     * @return FunctionScore
     */
    private function prepareSuggestNameQuery($search_for, User $user)
    {
        $query = new FunctionScore();
        $query->setBoostMode(FunctionScore::BOOST_MODE_SUM);
        $query->setParam('field_value_factor', ['field' => 'extra_score', 'modifier' => 'log1p']);

        $filter = self::prepareQueryFilter($user);

        if ($filter instanceof AbstractFilter) {
            $query->setFilter($filter);
        }

        $query->setQuery(new Prefix(['name_suggestions' => $search_for]));

        return $query;
    }

    /**
     * Return query filter.
     *
     * @param  User                         $user
     * @param  Criterion[]                  $criterions
     * @return \Elastica\Filter\BoolOr|null
     */
    private function prepareQueryFilter(User $user, $criterions = null)
    {
        $access_filter = $this->prepareUserAccessFilter($user);

        $specified_filter = $criterions && is_foreachable($criterions) ? $this->prepareUserSpecifiedFilter($criterions) : null;

        if ($access_filter instanceof AbstractFilter && $specified_filter instanceof AbstractFilter) {
            $and = new BoolFilter();
            $and->addMust($access_filter);
            $and->addMust($specified_filter);

            return $and;
        } elseif ($access_filter instanceof AbstractFilter) {
            return $access_filter;
        } elseif ($specified_filter instanceof AbstractFilter) {
            return $specified_filter;
        } else {
            return null;
        }
    }

    /**
     * Prepare query and data for auto-complete operation.
     *
     * @param  string $trigger
     * @param  string $search_for
     * @param  User   $user
     * @return array
     */
    private function prepareAutocompleteQuery($trigger, $search_for, User $user)
    {
        $suggesters = $this->getSpecialSuggesters();
        $field_name = $suggesters[$trigger]['field'];

        $filter = $this->prepareQueryFilter($user);

        if ($filter instanceof AbstractFilter) {
            $query = new FilteredQuery();
            $query->setFilter($filter);
        } else {
            $query = new Query();
        }

        $this->setSort($query, $field_name);

        $query->setQuery(new Prefix([$field_name => $search_for]));

        return [$query, $field_name, (isset($suggesters[$trigger]['also_fetch']) && is_foreachable($suggesters[$trigger]['also_fetch']) ? $suggesters[$trigger]['also_fetch'] : [])];
    }

    /**
     * Set sort for the given query.
     *
     * @param Query|FilteredQuery $query
     * @param string              $field_name
     * @param string              $order
     * @todo Remove when FiltedQuery gets setSort() method
     */
    private function setSort(&$query, $field_name, $order = 'asc')
    {
        if ($query instanceof Query) {
            $query->setSort([$field_name => ['order' => $order]]);
        } elseif ($query instanceof FilteredQuery) {
            //$query->setSort([ $field_name => [ 'order' => $order] ]);
        }
    }

    /**
     * Return trigger if we need to run a special suggester.
     *
     * @param  string      $search_for
     * @return string|null
     */
    private function shouldAutocomplete(&$search_for)
    {
        $suggesters = $this->getSpecialSuggesters();
        $trigger = substr_utf($search_for, 0, 1);

        if (isset($suggesters[$trigger]) && isset($suggesters[$trigger]['field'])) {
            $search_for = substr_utf($search_for, 1);

            return $trigger;
        }

        return null;
    }

    /**
     * @var array
     */
    private $special_suggesters = false;

    /**
     * Return a list of special suggesters.
     *
     * @return array
     */
    private function getSpecialSuggesters()
    {
        if ($this->special_suggesters === false) {
            $this->special_suggesters = [];
            Events::trigger('on_search_special_suggesters', [&$this->special_suggesters]);
        }

        return $this->special_suggesters;
    }

    /**
     * Prepare filter based on user's access permissions.
     *
     * @param  User        $user
     * @return BoolOr|null
     */
    private function prepareUserAccessFilter(User $user)
    {
        if (!$user->isOwner()) {
            $filter = new Filter();

            Events::trigger('on_user_access_search_filter', [&$user, &$filter]);

            if ($filter->hasRules()) {
                return $filter->getFilter();
            }
        }

        return null;
    }

    /**
     * @param  Criterion[]  $criterions
     * @return BoolAnd|null
     */
    private function prepareUserSpecifiedFilter($criterions = null)
    {
        $filter = new BoolAnd();

        foreach ($criterions as $criterion) {
            $value = (array) $criterion->getValue();

            switch (count($value)) {
                case 0:
                    break;
                case 1:
                    $filter->addFilter(new Term([$criterion->getField() => first($value)]));
                    break;
                case 2:
                    if ($criterion->getOperator() == Criterion::BETWEEN) {
                        $filter->addFilter(new Range($criterion->getField(), [
                            'gte' => $value[0] instanceof DateTimeValue ? $value[0]->toMySQL() : $value[0],
                            'lte' => $value[1] instanceof DateTimeValue ? $value[1]->toMySQL() : $value[1],
                        ]));
                    } else {
                        $filter->addFilter(new Terms($criterion->getField(), $value));
                    }

                    break;
                default:
                    $filter->addFilter(new Terms($criterion->getField(), $value));
            }
        }

        return count($filter->getFilters()) ? $filter : null;
    }

    /**
     * Suggest name.
     *
     * @param  string $search_for
     * @param  User   $user
     * @return array
     */
    abstract protected function suggestName($search_for, User $user);

    /**
     * Run suggester in auto-complete mode.
     *
     * @param  string $trigger
     * @param  string $search_for
     * @param  User   $user
     * @return array
     */
    abstract protected function autocomplete($trigger, $search_for, User $user);
}
