<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace Angie\Search\Adapter;

use ActiveCollab\ActiveCollabJobs\Jobs\Search\Autocomplete;
use ActiveCollab\ActiveCollabJobs\Jobs\Search\CreateIndex;
use ActiveCollab\ActiveCollabJobs\Jobs\Search\CreateMapping;
use ActiveCollab\ActiveCollabJobs\Jobs\Search\DeleteDocument;
use ActiveCollab\ActiveCollabJobs\Jobs\Search\DeleteIndex;
use ActiveCollab\ActiveCollabJobs\Jobs\Search\IndexDocument;
use ActiveCollab\ActiveCollabJobs\Jobs\Search\Query;
use ActiveCollab\ActiveCollabJobs\Jobs\Search\Suggest;
use ActiveCollab\JobsQueue\Jobs\Job;
use Angie\Search\Adapter;
use Angie\Search\Criterion;
use Angie\Search\Item;
use AngieApplication;
use Integrations;
use SearchIntegration;
use User;

/**
 * Queued search adapter.
 *
 * @package Angie\Search\Adapter
 */
class Queued extends Adapter
{
    use Adapter\ElasticSearch\PrepareQueries;

    /**
     * Set up the index.
     */
    public function setUp()
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return;
        }

        $job = new CreateIndex([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'priority' => Job::HAS_HIGHEST_PRIORITY,
            'number_of_shards' => 2,
            'number_of_replicas' => 1,
        ]);

        if (php_sapi_name() == 'cli') {
            AngieApplication::jobs()->execute($job);
        } else {
            AngieApplication::jobs()->dispatch($job, SearchIntegration::JOBS_QUEUE_CHANNEL);
        }
    }

    /**
     * Destroy the index.
     */
    public function tearDown()
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return;
        }

        $job = new DeleteIndex([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'priority' => Job::HAS_HIGHEST_PRIORITY,
        ]);

        if (php_sapi_name() == 'cli') {
            AngieApplication::jobs()->execute($job);
        } else {
            AngieApplication::jobs()->dispatch($job, SearchIntegration::JOBS_QUEUE_CHANNEL);
        }
    }

    /**
     * Initialize index for the specific type.
     *
     * @param string|array $type
     * @param array|null   $fields
     */
    public function initializeType($type, $fields = null)
    {
        if (is_array($type)) {
            foreach ($type as $k => $v) {
                $this->createMapping($k, $v);
            }
        } else {
            $this->createMapping($type, $fields);
        }
    }

    /**
     * Create a mapping for the given type (if mapping does not exist).
     *
     * @param string $type
     * @param array  $fields
     */
    private function createMapping($type, $fields)
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return;
        }

        $job = new CreateMapping([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'type' => $type,
            'fields' => $fields,
            'priority' => Job::HAS_PRIORITY,
        ]);

        AngieApplication::jobs()->dispatch($job, SearchIntegration::JOBS_QUEUE_CHANNEL);
    }

    /**
     * Return indexed record.
     *
     * @param  Item       $item
     * @return array|null
     */
    public function get(Item $item)
    {
    }

    /**
     * Add an item to the index.
     *
     * @param Item $item
     * @param bool $bulk
     */
    public function add(Item $item, $bulk = false)
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return;
        }

        AngieApplication::jobs()->dispatch(new IndexDocument([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'type' => $item->getSearchIndexType(),
            'id' => $item->getId(),
            'fields' => $item->searchSerialize(),
            'attempts' => 5,
            'delay' => 60,
            'first_attempt_delay' => 0,
        ]), SearchIntegration::JOBS_QUEUE_CHANNEL);
    }

    /**
     * Update an item.
     *
     * @param Item $item
     * @param bool $bulk
     */
    public function update(Item $item, $bulk = false)
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return;
        }

        AngieApplication::jobs()->dispatch(new IndexDocument([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'type' => $item->getSearchIndexType(),
            'id' => $item->getId(),
            'fields' => $item->searchSerialize(),
            'attempts' => 5,
            'delay' => 60,
            'first_attempt_delay' => 0,
        ]), SearchIntegration::JOBS_QUEUE_CHANNEL);
    }

    /**
     * Remove an item.
     *
     * @param Item $item
     * @param bool $bulk
     */
    public function remove(Item $item, $bulk = false)
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return;
        }

        AngieApplication::jobs()->dispatch(new DeleteDocument([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'type' => $item->getSearchIndexType(),
            'id' => $item->getId(),
            'attempts' => 5,
            'delay' => 60,
            'first_attempt_delay' => 0,
        ]), SearchIntegration::JOBS_QUEUE_CHANNEL);
    }

    /**
     * Query the index.
     *
     * @param  string      $search_for
     * @param  User        $user
     * @param  Criterion[] $criterions
     * @param  int         $page
     * @return array
     */
    public function query($search_for, User $user, $criterions = null, $page = 1)
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return [];
        }

        return AngieApplication::jobs()->execute(new Query([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'query' => $this->prepareQuery($search_for, $user, $criterions, $page)->toArray(),
        ]));
    }

    /**
     * Suggest name.
     *
     * @param  string $search_for
     * @param  User   $user
     * @return array
     */
    protected function suggestName($search_for, User $user)
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return [];
        }

        return AngieApplication::jobs()->execute(new Suggest([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'query' => $this->prepareSuggestNameQuery($search_for, $user)->toArray(),
        ]));
    }

    /**
     * Run suggester in auto-complete mode.
     *
     * @param  string $trigger
     * @param  string $search_for
     * @param  User   $user
     * @return array
     */
    protected function autocomplete($trigger, $search_for, User $user)
    {
        list($host, $port) = $this->getHostAndPort();

        $result = [];

        if (empty($host) || empty($search_for)) {
            return $result;
        }

        /** @var \Elastica\Query $query */
        list($query, $field_name, $also_fetch_fields) = $this->prepareAutocompleteQuery($trigger, $search_for, $user);

        return AngieApplication::jobs()->execute(new Autocomplete([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'query' => $query->toArray(),
            'field_name' => $field_name,
            'also_fetch_fields' => $also_fetch_fields,
        ]));
    }

    /**
     * Start batch import of the data.
     *
     * @param string $type
     * @param int    $items_per_iteration
     */
    public function batchStart($type, $items_per_iteration = 500)
    {
    }

    /**
     * Add record to the index.
     *
     * @param Item $item
     */
    public function batchAdd($item)
    {
        list($host, $port) = $this->getHostAndPort();

        if (empty($host)) {
            return;
        }

        AngieApplication::jobs()->dispatch(new IndexDocument([
            'instance_id' => AngieApplication::getAccountId(),
            'host' => $host,
            'port' => $port,
            'index' => $this->getIndexName(),
            'type' => $item->getSearchIndexType(),
            'id' => $item->getId(),
            'fields' => $item->searchSerialize(),
            'attempts' => 5,
            'delay' => 60,
            'first_attempt_delay' => 0,
        ]), SearchIntegration::JOBS_QUEUE_CHANNEL);
    }

    /**
     * Finished with the batch import.
     *
     * @param string $type
     */
    public function batchDone($type)
    {
    }

    /**
     * @var string|null
     */
    private $host = false;

    /**
     * @var int|null
     */
    private $port = false;

    /**
     * Return connection host and port.
     *
     * @return array
     */
    private function getHostAndPort()
    {
        if ($this->host === false && $this->port === false) {
            if (AngieApplication::isOnDemand() && defined('ELASTIC_SEARCH_HOSTS') && ELASTIC_SEARCH_HOSTS) {
                foreach (explode("\n", ELASTIC_SEARCH_HOSTS) as $host) {
                    if ($host = trim($host)) {
                        $parts = parse_url($host);

                        $this->host = isset($parts['host']) ? $parts['host'] : 'localhost';
                        $this->port = isset($parts['port']) ? $parts['port'] : 9200;

                        break;
                    }
                }

                if (empty($this->host) && empty($this->port)) {
                    $this->host = '';
                    $this->port = 9200;
                }
            } else {
                /** @var SearchIntegration $integration */
                $integration = Integrations::findFirstByType('SearchIntegration');

                $this->host = (string) $integration->getHost();
                $this->port = (int) $integration->getPort();
            }
        }

        return [$this->host, $this->port];
    }
}
