<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace Angie\Search\Item;

use Angie\Search;
use Angie\Search\Item;
use InvalidParamError;
use ITrash;

/**
 * Base search item implementation.
 *
 * @package Angie\Search
 */
trait Implementation
{
    /**
     * @var bool
     */
    private $update_search_on_next_save = true;

    /**
     * Say hello to the parent object.
     */
    public function AngieSearchItemImplementation()
    {
        $this->registerEventHandler('on_after_save', function ($is_new, $modifications) {
            if ($this->update_search_on_next_save) {
                if ($is_new) {
                    Search::add($this);
                } elseif (!empty($modifications) && $this->shouldSearchIndexBeUpdated($modifications)) {
                    Search::update($this);
                }
            } else {
                $this->update_search_on_next_save = true;
            }
        });

        $this->registerEventHandler('on_after_delete', function ($bulk) {
            Search::remove($this, $bulk);
        });

        if ($this instanceof ITrash) {
            $this->registerEventHandler('on_after_move_to_trash', function ($bulk) {
                Search::remove($this, $bulk);
            });

            $this->registerEventHandler('on_after_restore_from_trash', function ($bulk) {
                Search::add($this, $bulk);
            });
        }
    }

    /**
     * Return true if we should update the index.
     *
     * @param  array $modifications
     * @return bool
     */
    private function shouldSearchIndexBeUpdated($modifications)
    {
        if ($this instanceof ITrash && !empty($modifications['is_trashed'])) {
            return false; // Let the trash events handle the index refresh
        }

        return count(array_intersect($this->getSearchFieldNames(), array_keys($modifications))) > 0;
    }

    /**
     * @var array
     */
    private $search_fields = false, $search_field_names = false;

    /**
     * Return names of search fields.
     *
     * @return array
     */
    private function getSearchFieldNames()
    {
        if ($this->search_field_names === false) {
            $this->search_field_names = array_keys($this->getSearchFields());
        }

        return $this->search_field_names;
    }

    /**
     * Return a list of fields that are indexed for this type.
     *
     * @return array
     */
    public function getSearchFields()
    {
        if ($this->search_fields === false) {
            $this->search_fields = [];
            $this->triggerEvent('on_search_fields', [&$this->search_fields]);
        }

        return $this->search_fields;
    }

    /**
     * @return int
     */
    protected function getSearchExtraScore()
    {
        $extra_score = 0;
        $this->triggerEvent('on_search_extra_score', [&$extra_score]);

        return $extra_score;
    }

    /**
     * Serialize object to be indexed.
     *
     * @return array
     */
    public function searchSerialize()
    {
        $result = [
            'id' => $this->getId(),
            'type' => $this->getModelName(false, true),
            'name' => $this->getName(),
            'url' => $this->getUrlPath(),
            'extra_score' => $this->getSearchExtraScore(),
        ];

        $result['name_sortable'] = $result['name'];
        $result['name_suggestions'] = $result['name'];

        $this->triggerEvent('on_search_serialize', [&$result]);

        return $result;
    }

    /**
     * Return type under which it this object is stored in the search index.
     *
     * @return string
     */
    public function getSearchIndexType()
    {
        return $this->getModelName(false, true);
    }

    /**
     * Call this method if you do not want to update the search record on the next save call.
     *
     * This is useful if you are creating an object (invoice) with related objects that are going to be added after
     * $this is saved and are relevant for the serach (invoice items)
     */
    public function dontUpdateSearchIndexOnNextSave()
    {
        $this->update_search_on_next_save = false;
    }

    // ---------------------------------------------------
    //  Expectations
    // ---------------------------------------------------

    /**
     * Return object ID.
     *
     * @return int
     */
    abstract public function getId();

    /**
     * Return name.
     *
     * @return string
     */
    abstract public function getName();

    /**
     * Return name of this model.
     *
     * @param  bool   $underscore
     * @param  bool   $singular
     * @return string
     */
    abstract public function getModelName($underscore = false, $singular = false);

    /**
     * Return object path.
     *
     * @return string
     */
    abstract public function getObjectPath();

    /**
     * Register an internal event handler.
     *
     * @param $event
     * @param $handler
     * @throws InvalidParamError
     */
    abstract protected function registerEventHandler($event, $handler);

    /**
     * Trigger an internal event.
     *
     * @param string $event
     * @param array  $event_parameters
     */
    abstract protected function triggerEvent($event, $event_parameters = null);

    /**
     * Return url path.
     *
     * @return string
     */
    abstract public function getUrlPath();
}
