<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use Angie\Events;
use Angie\Http\Request;
use Angie\Http\Response;

AngieApplication::useController('auth_required', EnvironmentFramework::INJECT_INTO);

/**
 * Feedback controller.
 *
 * @package ActiveCollab.modules.system
 * @subpackage controllers
 */
class FeedbackController extends AuthRequiredController
{
    /**
     * Send feedback to the team.
     *
     * @param  Request                  $request
     * @param  User                     $user
     * @return FeedbackNotification|int
     */
    public function send(Request $request, User $user)
    {
        $comment = trim($request->post('comment'));

        if (empty($comment)) {
            return Response::BAD_REQUEST;
        }

        $details = $request->post('details');

        if (!is_array($details)) {
            $details = [];
        }

        if (!empty($details['sender_name'])) {
            $details['sender_name'] .= ' (' . get_class($user) . ')';
        } else {
            $details['user_role'] = get_class($user);
        }

        if (!empty($details['user_agent'])) {
            require_once APPLICATION_PATH . '/vendor/donatj/phpuseragentparser/Source/UserAgentParser.php';
            $details['user_agent'] = implode(' ', parse_user_agent($details['user_agent']));
        }

        if (AngieApplication::isOnDemand()) {
            $details['account_id'] = (int) AngieApplication::getAccountId();

            if (defined('ON_DEMAND_ACCOUNT_STATUS')) {
                switch (ON_DEMAND_ACCOUNT_STATUS) {
                    case OnDemand::STATUS_PENDING_ACTIVATION:
                        $details['account_status'] = 'Pending Activation';
                        break;
                    case OnDemand::STATUS_ACTIVE_FREE:
                        $details['account_status'] = 'Free (active)';
                        break;
                    case OnDemand::STATUS_ACTIVE_PAID:
                        $details['account_status'] = 'Paid (active)';
                        break;
                    case OnDemand::STATUS_FAILED_PAYMENT:
                        $details['account_status'] = 'Paid (failed payment)';
                        break;
                    case OnDemand::STATUS_SUSPENDED_FREE:
                        $details['account_status'] = 'Free (suspended)';
                        break;
                    case OnDemand::STATUS_SUSPENDED_PAID:
                        $details['account_status'] = 'Paid (suspended)';
                        break;
                    case OnDemand::STATUS_RETIRED_FREE:
                        $details['account_status'] = 'Free (retired)';
                        break;
                    case OnDemand::STATUS_RETIRED_PAID:
                        $details['account_status'] = 'Paid (retired)';
                        break;
                    case OnDemand::STATUS_PENDING_DELETION:
                        $details['account_status'] = 'Pending Deletion';
                        break;
                }
            }

            if (defined('ON_DEMAND_PLAN_NAME') && defined('ON_DEMAND_PLAN_PERIOD')) {
                $details['account_plan'] = ON_DEMAND_PLAN_NAME . ', ' . ON_DEMAND_PLAN_PERIOD;

                if (defined('ON_DEMAND_ACCOUNT_STATUS_EXPIRES_ON') && strpos(ON_DEMAND_ACCOUNT_STATUS_EXPIRES_ON, ' ') !== false) {
                    $details['account_plan'] .= ' (' . first(explode(' ', ON_DEMAND_ACCOUNT_STATUS_EXPIRES_ON)) . ')';
                }
            }
        } else {
            $details['license_key'] = strip_tags(trim(AngieApplication::getLicenseKey()));

            $status = $this->getSystemStatus();

            if (empty($status) || !is_array($status)) {
                $details['system_status'] = 'Failed to load';
            } else {
                foreach (['cron', 'search', 'email'] as $check) {
                    $is_ok_key = "{$check}_is_ok";

                    if (array_key_exists($is_ok_key, $status)) {
                        if ($status[$is_ok_key]) {
                            $details[$is_ok_key] = true;
                        } else {
                            $errors_key = "{$check}_errors";

                            if (array_key_exists($errors_key, $status) && is_array($status[$errors_key])) {
                                $details[$is_ok_key] = 'No. Errors: ' . implode(', ', $status[$errors_key]);
                            } else {
                                $details[$is_ok_key] = 'No. Error details are empty!';
                            }
                        }
                    }
                }
            }
        }

        /** @var FeedbackNotification $notification */
        $notification = AngieApplication::notifications()->notifyAbout('system/feedback', null, $user);

        return $notification
            ->setComment($comment)
            ->setDetails($details)
            ->sendToUsers(new AnonymousUser('ActiveCollab Support', 'support@activecollab.com'), true);
    }

    /**
     * @return array
     */
    public function check()
    {
        if (AngieApplication::isOnDemand()) {
            return ['is_ok' => true, 'errors' => []];
        } else {
            $status = $this->getSystemStatus();

            return [
                'is_ok' => !empty($status['email_is_ok']),
                'errors' => !empty($status['email_errors']) ? $status['email_errors'] : [],
            ];
        }
    }

    /**
     * @return array
     */
    private function getSystemStatus()
    {
        $status = [];
        Events::trigger('on_system_status', [&$status]);

        return $status;
    }
}
