<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use ActiveCollab\Authentication\Adapter\AdapterInterface;
use ActiveCollab\Authentication\Adapter\BrowserSessionAdapter;
use ActiveCollab\Authentication\Adapter\TokenBearerAdapter;
use ActiveCollab\Authentication\AuthenticationResult\AuthenticationResultInterface;
use ActiveCollab\Authentication\AuthenticationResult\Transport\TransportInterface;
use ActiveCollab\Authentication\Authorizer\ExceptionAware\ExceptionAwareInterface as ExceptionAwareAuthorizerInterface;
use ActiveCollab\Authentication\Authorizer\RequestAware\RequestAwareInterface as RequestAwareAuthorizerInterface;
use Angie\Http\Request;
use Angie\Http\Response;
use Angie\Http\Response\StatusResponse\StatusResponseInterface;

AngieApplication::useController('auth_not_required', SystemModule::NAME);

/**
 * Application level user session controller.
 *
 * @package activeCollab.modules.system
 * @subpackage controllers
 */
class UserSessionController extends AuthNotRequiredController
{
    /**
     * @param  Request                       $request
     * @param  User                          $user
     * @return array|StatusResponseInterface
     */
    public function who_am_i(Request $request, User $user = null)
    {
        if ($user instanceof User) {
            return Users::prepareCollection('initial_for_logged_user', $user);
        } else {
            return $this->requireAuthorization();
        }
    }

    /**
     * Log user in based on provided data.
     *
     * @param  Request                                    $request
     * @return TransportInterface|StatusResponseInterface
     * @throws Exception
     */
    public function login(Request $request)
    {
        $authorizer = AngieApplication::authentication()->getAuthorizer();

        $credentials = $request->post();
        $payload = null;

        try {
            if ($authorizer instanceof RequestAwareAuthorizerInterface && $authorizer->getRequestProcessor()) {
                $request_processing_result = $authorizer->getRequestProcessor()->processRequest($request);

                $credentials = $request_processing_result->getCredentials();
                $payload = $request_processing_result->getDefaultPayload();
            }

            return AngieApplication::authentication()->authorizeForAdapter($credentials, BrowserSessionAdapter::class, $payload);
        } catch (Exception $e) {
            AngieApplication::log()->notice('User {username} failed to log in. Reason: {reason}.', [
                'username' => array_var($credentials, 'username', '--username not set--'),
                'reason' => $e->getMessage(),
                'exception' => $e,
            ]);

            if ($authorizer instanceof ExceptionAwareAuthorizerInterface) {
                $exception_handling_result = $authorizer->handleException($credentials, $e);

                if ($exception_handling_result !== null) {
                    return $exception_handling_result;
                }
            }

            throw($e);
        }
    }

    /**
     * Kill the current user session.
     *
     * @param  Request                $request
     * @param  User|null              $user
     * @return TransportInterface|int
     */
    public function logout(Request $request, User $user = null)
    {
        if ($user) {
            /** @var AdapterInterface $authentication_adapter */
            $authentication_adapter = $request->getAttribute('authentication_adapter');

            /** @var AuthenticationResultInterface $authenticated_with */
            $authenticated_with = $request->getAttribute('authenticated_with');

            if ($authentication_adapter && $authenticated_with) {
                return AngieApplication::authentication()->terminate($authentication_adapter, $authenticated_with);
            }
        }

        return Response::BAD_REQUEST;
    }

    /**
     * Issue token.
     *
     * @param  Request                $request
     * @return TransportInterface|int
     */
    public function issue_token(Request $request)
    {
        try {
            return AngieApplication::authentication()->authorizeForAdapter($request->post(), TokenBearerAdapter::class);
        } catch (Exception $e) {
            AngieApplication::log()->notice('User {username} failed to log in. Reason: {reason}.', [
                'username' => array_var($credentials, 'username', '--username not set--'),
                'reason' => $e->getMessage(),
                'exception' => $e,
            ]);

            return Response::OPERATION_FAILED;
        }
    }

    /**
     * View invitation.
     *
     * @param  Request   $request
     * @return array|int
     */
    public function view_invitation(Request $request)
    {
        $invitation = $this->getInvitationFromParameters($request);

        if ($invitation instanceof UserInvitation) {
            $user = $invitation->getUser();
            $invited_by = $invitation->getCreatedBy();
            $invited_to = $invitation->getInvitedTo();

            return [
                'status' => $invitation->getStatus(),
                'user' => $user instanceof User && $user->isActive() ? ['id' => $user->getId(), 'email' => $user->getEmail(), 'first_name' => $user->getFirstName(), 'last_name' => $user->getLastName()] : null,
                'invited_by' => $invited_by instanceof User ? ['id' => $invited_by->getId(), 'full_name' => $invited_by->getDisplayName(), 'first_name' => $invited_by->getFirstName(), 'last_name' => $invited_by->getLastName()] : null,
                'invited_to' => $invited_to ? ['id' => $invited_to->getId(), 'class' => get_class($invited_to), 'name' => $invited_to->getName()] : null,
            ];
        }

        return $invitation;
    }

    /**
     * @param  Request                $request
     * @return TransportInterface|int
     */
    public function accept_invitation(Request $request)
    {
        if ($invitation = $this->getInvitationFromParameters($request)) {
            if ($invitation->getStatus() === UserInvitation::ACCEPTABLE) {
                list($first_name, $last_name, $password, $language_id, $uploaded_avatar_code) = $this->validateInvitationParameters($request);

                $invited_user = $invitation->getUser();

                Users::update($invited_user, [
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'password' => $password,
                    'language_id' => $language_id >= 1 && Languages::findById($language_id) instanceof Language ? $language_id : 0,
                    'uploaded_avatar_code' => $uploaded_avatar_code,
                ]);

                $invited_user->invitationAccepted();

                return AngieApplication::authentication()->authorizeForAdapter([
                    'username' => $invited_user->getEmail(),
                    'password' => $password,
                ], BrowserSessionAdapter::class, Users::prepareCollection('initial_for_logged_user', $invited_user));
            }

            return Response::GONE; // Expired
        }

        return Response::NOT_FOUND;
    }

    /**
     * Return response that will require user to authorize.
     *
     * @return StatusResponseInterface
     */
    private function requireAuthorization()
    {
        return new Angie\Http\Response\StatusResponse\StatusResponse(401, 'User not authenticated.', Users::prepareCollection('initial_for_logged_user', null));
    }

    /**
     * Return a valid invitation or proper status code (not found or expired).
     *
     * @param  Request            $request
     * @return UserInvitation|int
     */
    private function getInvitationFromParameters(Request $request)
    {
        $invitation = UserInvitations::findByUserIdAndCode($request->get('user_id'), $request->get('code'));

        if ($invitation instanceof UserInvitation) {
            return $invitation;
        }

        return Response::NOT_FOUND;
    }

    /**
     * Read, validate and return invitation accept parameters.
     *
     * @param  Request          $request
     * @return array
     * @throws ValidationErrors
     */
    private function validateInvitationParameters(Request $request)
    {
        $first_name = trim($request->post('first_name'));
        $last_name = trim($request->post('last_name'));
        $password = $request->post('password');
        $language_id = (int) $request->post('language_id');
        $uploaded_avatar_code = $request->post('uploaded_avatar_code');

        $errors = new ValidationErrors();

        if (empty($first_name)) {
            $errors->fieldValueIsRequired('first_name');
        }

        if (empty($last_name)) {
            $errors->fieldValueIsRequired('last_name');
        }

        if (trim($password) === '') {
            $errors->fieldValueIsRequired('password');
        }

        if (empty($language_id)) {
            $errors->fieldValueIsRequired('language');
        }

        if ($errors->hasErrors()) {
            throw $errors;
        }

        return [$first_name, $last_name, $password, $language_id, $uploaded_avatar_code];
    }
}
