<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace ActiveCollab\ActiveCollabJobs\Jobs\Shepherd;

use ActiveCollab\ActiveCollabJobs\ExecuteCliCommand\ExecuteCliCommandTrait;
use ActiveCollab\ActiveCollabJobs\Jobs\Job;
use ActiveCollab\ConfigFile\ConfigFile;
use Psr\Log\LoggerInterface;
use RuntimeException;

/**
 * @package ActiveCollab\ActiveCollabJobs\Jobs\Instance
 */
class ExecuteShephedCliCommand extends Job
{
    use ExecuteCliCommandTrait;

    /**
     * @var string
     */
    private $shepherd_version;

    /**
     * Clean up work folder for the given instance.
     *
     * @throws RuntimeException
     */
    public function execute()
    {
        $shepherd_run_php_path = $this->getShepherdRunPhpPath();

        if (!is_file($shepherd_run_php_path)) {
            throw new RuntimeException("Failed to find Shepherd CLI script at '$shepherd_run_php_path'");
        }

        $output = [];
        $code = 0;

        $command_to_execute = "cd /var/www/shepherd && php $shepherd_run_php_path " . $this->prepareCommandFromData();

        exec($command_to_execute, $output, $code);

        $this->logCommandExecution($command_to_execute, $output, $code);

        if ($code === 0) {
            return;
        } else {
            throw new RuntimeException('Failed to execute command. Output: ' . implode("\n", $output), $code);
        }
    }

    /**
     * Log command execution.
     *
     * @param string $command
     * @param array  $output
     * @param int    $code
     */
    private function logCommandExecution($command, $output, $code)
    {
        $logger = $this->hasContainer() && $this->getContainer()->has('log') ?
            $this->getContainer()->get('log') :
            $this->getLog();

        if ($logger instanceof LoggerInterface) {
            $logger->event('shepherd_command_executed', 'Shepherd CLI command executed', [
                'command' => $command,
                'exit_code' => $code,
                'output' => is_array($output) ? implode("\n", $output) : (string) $output,
            ]);
        }
    }

    /**
     * Return path to Shepherd's run.php.
     *
     * @return string
     */
    public function getShepherdRunPhpPath()
    {
        return '/var/www/shepherd/app/' . $this->getShepherdVersion() . '/angie/tools/cli/run.php';
    }

    /**
     * Return currently used Shepherd version.
     *
     * @return string
     */
    private function getShepherdVersion()
    {
        if (empty($this->shepherd_version)) {
            $shepherd_version_file = new ConfigFile('/var/www/shepherd/config/version.php');
            $shepherd_version = $shepherd_version_file->getOption('APPLICATION_VERSION');

            if ($shepherd_version) {
                $this->shepherd_version = $shepherd_version;
            } else {
                throw new RuntimeException("Can't read Shepherd version");
            }
        }

        return $this->shepherd_version;
    }
}
