<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace ActiveCollab\Command;

use Angie\Command\Command;
use Angie\Search;
use DB;
use Exception;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * @package Angie\Command
 */
final class RebuildSearchIndexCommand extends Command
{
    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        parent::configure();

        $this->setDescription('Rebuild search index')
                ->addOption('skip-tasks', '', InputOption::VALUE_NONE, 'Skip tasks');
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $output->writeln('<info>OK:</info> Rebuilding search');

        $microtime = microtime(true);

        try {
            DB::execute('DELETE FROM jobs_queue WHERE type IN (?)', [
                'ActiveCollab\\ActiveCollabJobs\\Jobs\\Search\\Query',
                'ActiveCollab\\ActiveCollabJobs\\Jobs\\Search\\CreateIndex',
                'ActiveCollab\\ActiveCollabJobs\\Jobs\\Search\\DeleteIndex',
                'ActiveCollab\\ActiveCollabJobs\\Jobs\\Search\\CreateMapping',
                'ActiveCollab\\ActiveCollabJobs\\Jobs\\Search\\IndexDocument',
                'ActiveCollab\\ActiveCollabJobs\\Jobs\\Search\\DeleteDocument',
            ]); // Remove all existing search tasks from jobs queue

            Search::reset();

            $output->writeln('<info>OK:</info> Sleeping for a second...');
            sleep(1);

            $skip_tasks = $input->getOption('skip-tasks');

            foreach (Search::getRebuildActions() as $action_name => $action) {
                if ($skip_tasks && $action_name == 'rebuild_project_tasks') {
                    continue;
                }

                if (is_callable($action['callback'])) {
                    $output->writeln("<info>OK:</info> $action[label]");
                    call_user_func($action['callback']);
                } else {
                    return $this->abort("Rebuild action '$action_name' has no proper execution method", 1, $input, $output);
                }
            }
        } catch (Exception $e) {
            return $this->abortDueToException($e, $input, $output);
        }

        return $this->success('Done in ' . round(microtime(true) - $microtime, 2) . 's', $input, $output);
    }
}
