<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace ActiveCollab\Command\User;

use AngieApplication;
use ConfigOptionDnxError;
use ConfigOptions;
use DB;
use Exception;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use UploadedFile;
use UploadedFiles;
use User;
use Users;

/**
 * @package ActiveCollab\Command
 */
class EditCommand extends UserCommand
{
    /**
     * {@inheritdoc}
     */
    public function configure()
    {
        parent::configure();

        $this->setDescription('Update a user account')
            ->addArgument('user_id', InputArgument::REQUIRED)
            ->addArgument('email_address', InputArgument::REQUIRED)
            ->addArgument('first_name', InputArgument::OPTIONAL)
            ->addArgument('last_name', InputArgument::OPTIONAL)
            ->addOption('options', '', InputOption::VALUE_REQUIRED, 'JSON serialized array of options')
            ->addOption('avatar-path', '', InputOption::VALUE_REQUIRED);
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $user_id = (int) $input->getArgument('user_id');
        $email_address = $input->getArgument('email_address');

        $first_name = $input->getArgument('first_name');
        $last_name = $input->getArgument('last_name');

        $options = $input->getOption('options');
        $avatar_path = $input->getOption('avatar-path');

        if (!is_valid_email($email_address)) {
            return $this->abort('Email must be valid email address', 1, $input, $output);
        }

        /** @var User $user */
        $user = Users::findById($user_id);
        if (empty($user)) {
            return $this->abort("User with id #{$user_id} not found", 1, $input, $output);
        }

        DB::transact(function () use ($user, $first_name, $last_name, $email_address, $options) {
            $user->setFirstName($first_name);
            $user->setLastName($last_name);
            $user->setEmail($email_address);

            $options = json_decode($options, true);

            if (!empty($options) && is_array($options)) {
                $valid_user_options = [
                    'format_date',
                    'format_time',
                    'time_timezone',
                    'time_first_week_day',
                ];

                if (isset($options['phone_mobile'])) {
                    unset($options['phone_mobile']);
                }

                foreach ($options as $option_name => $option_value) {
                    if (in_array($option_name, $valid_user_options) && ConfigOptions::exists($option_name)) {
                        try {
                            ConfigOptions::setValueFor($option_name, $user, $option_value);
                        } catch (ConfigOptionDnxError $e) {
                        }
                    } else {
                        $user->setFieldValue($option_name, $option_value);
                    }
                }
            }

            if ($user->isModified()) {
                $user->save();
            }
        });

        try {
            if (is_file($avatar_path)) {
                $current_avatar_location = $user->getAvatarLocation();

                list($target_path, $location) = AngieApplication::storeFile($avatar_path);

                $uploaded_file = UploadedFiles::create([
                    'name' => basename($avatar_path),
                    'mime_type' => 'image/png',
                    'size' => filesize($avatar_path),
                    'location' => $location,
                    'md5' => md5_file($target_path),
                ]);

                if ($uploaded_file instanceof UploadedFile) {
                    $user->setAvatarLocation($uploaded_file->getLocation());

                    AngieApplication::removeStoredFile($current_avatar_location);

                    $uploaded_file->keepFileOnDelete(true);
                    $uploaded_file->delete();
                }
            } else {
                //                $user->setAvatarLocation(null); // "reset to default" from aC ID
            }

            $user->save();

            return $this->success('Done', $input, $output);
        } catch (Exception $e) {
            return $this->abortDueToException($e, $input, $output);
        }
    }
}
