<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use Angie\Http\Request;
use Angie\Http\Response;

AngieApplication::useController('auth_required', EnvironmentFramework::INJECT_INTO);

/**
 * Integrations controller.
 *
 * @package angie.frameworks.environment
 * @subpackage controllers
 */
abstract class FwIntegrationsController extends AuthRequiredController
{
    /**
     * Selected integration instance.
     *
     * @var Integration
     */
    protected $active_integration;

    /**
     * {@inheritdoc}
     */
    protected function __before(Request $request, $user)
    {
        $before_result = parent::__before($request, $user);

        if ($before_result !== null) {
            return $before_result;
        }

        $this->active_integration = DataObjectPool::get('Integration', $request->getId('integration_id'));

        if ($this->active_integration instanceof Integration && !$this->active_integration->canView($user)) {
            return Response::NOT_FOUND;
        }
    }

    /**
     * Add new integration.
     *
     * @param Request $request
     * @param User    $user
     *
     * @return DataObject|int
     */
    public function add(Request $request, User $user)
    {
        return $user->isOwner() ? Integrations::create($request->post()) : Response::NOT_FOUND;
    }

    /**
     * Return integrations that the given user can see.
     *
     * @param  Request         $request
     * @param  User            $user
     * @return ModelCollection
     */
    public function index(Request $request, User $user)
    {
        return Integrations::getFor($user);
    }

    /**
     * Return selected integration.
     */
    public function view()
    {
        return $this->active_integration instanceof Integration ? $this->active_integration : Response::NOT_FOUND;
    }

    /**
     * Edit integration.
     *
     * @param Request $request
     * @param User    $user
     *
     * @return DataObject|int
     */
    public function edit(Request $request, User $user)
    {
        return $this->active_integration instanceof Integration && $this->active_integration->canEdit($user) ? Integrations::update($this->active_integration, $request->put()) : Response::NOT_FOUND;
    }

    /**
     * Drop the given integration.
     *
     * @param  Request  $request
     * @param  User     $user
     * @return bool|int
     */
    public function delete(Request $request, User $user)
    {
        return $this->active_integration instanceof Integration && $this->active_integration->canDelete($user) ? DataFilters::scrap($this->active_integration) : Response::NOT_FOUND;
    }
}
