<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace Angie\Authentication;

use AngieApplication;
use DateValue;
use DB;
use User;

/**
 * Security logs.
 *
 * @package ActiveCollab.modules.system
 * @subpackage models
 */
final class SecurityLogs
{
    const LOGIN = 'login';
    const INVALID_PASSWORD = 'failed';
    const EXPIRED = 'expired';
    const LOGOUT = 'logout';

    /**
     * Write security log.
     *
     * @param User      $user
     * @param string    $event
     * @param User|null $someone
     */
    public static function write(User $user, $event, $someone = null)
    {
        switch ($event) {
            case self::LOGIN:
                if ($someone instanceof User) {
                    DB::execute('INSERT INTO security_logs (user_id, user_name, user_email, user_agent, login_as_id, login_as_name, login_as_email, event, event_on, user_ip) VALUES (?, ?, ?, ?, ?, ?, ?, ?, UTC_TIMESTAMP(), ?)', $user->getId(), $user->getDisplayName(), $user->getEmail(), AngieApplication::getVisitorUserAgent(), $someone->getId(), $someone->getName(), $someone->getEmail(), $event, AngieApplication::getVisitorIp());
                } else {
                    DB::execute('INSERT INTO security_logs (user_id, user_name, user_email, user_agent, event, event_on, user_ip) VALUES (?, ?, ?, ?, ?, UTC_TIMESTAMP(), ?)', $user->getId(), $user->getDisplayName(), $user->getEmail(), AngieApplication::getVisitorUserAgent(), $event, AngieApplication::getVisitorIp());
                }
                break;

            case self::LOGOUT:
                if ($someone instanceof User) {
                    DB::execute('INSERT INTO security_logs (user_id, user_name, user_email, logout_by_id, logout_by_name, logout_by_email, user_agent, event, event_on, user_ip) VALUES (?, ?, ?, ?, ?, ?, ?, ?, UTC_TIMESTAMP(), ?)', $user->getId(), $user->getDisplayName(), $user->getEmail(), $someone->getId(), $someone->getName(), $someone->getEmail(), AngieApplication::getVisitorUserAgent(), $event, AngieApplication::getVisitorIp());
                } else {
                    DB::execute('INSERT INTO security_logs (user_id, user_name, user_email, user_agent, event, event_on, user_ip) VALUES (?, ?, ?, ?, ?, UTC_TIMESTAMP(), ?)', $user->getId(), $user->getDisplayName(), $user->getEmail(), AngieApplication::getVisitorUserAgent(), $event, AngieApplication::getVisitorIp());
                }

                break;

            case self::INVALID_PASSWORD:
                if ($someone instanceof User) {
                    DB::execute('INSERT INTO security_logs (user_id, user_name, user_email, user_agent, login_as_id, login_as_name, login_as_email, event, event_on, user_ip) VALUES (?, ?, ?, ?, ?, ?, ?, ?, UTC_TIMESTAMP(), ?)', $user->getId(), $user->getDisplayName(), $user->getEmail(), AngieApplication::getVisitorUserAgent(), $someone->getId(), $someone->getName(), $someone->getEmail(), $event, AngieApplication::getVisitorIp());
                } else {
                    DB::execute('INSERT INTO security_logs (user_id, user_name, user_email, user_agent, event, event_on, user_ip) VALUES (?, ?, ?, ?, ?, UTC_TIMESTAMP(), ?)', $user->getId(), $user->getDisplayName(), $user->getEmail(), AngieApplication::getVisitorUserAgent(), $event, AngieApplication::getVisitorIp());
                }

                self::triggerFirewall($user);
                break;

            default:
                DB::execute('INSERT INTO security_logs (user_id, user_name, user_email, user_agent, event, event_on, user_ip) VALUES (?, ?, ?, ?, ?, UTC_TIMESTAMP(), ?)', $user->getId(), $user->getDisplayName(), $user->getEmail(), AngieApplication::getVisitorUserAgent(), self::EXPIRED, AngieApplication::getVisitorIp());
        }
    }

    /**
     * Log security for login try.
     *
     * @param string $email
     */
    public static function logAttempt($email = null)
    {
        DB::execute('INSERT INTO security_logs (user_email, user_agent, event, event_on, user_ip) VALUES (?, ?, ?, UTC_TIMESTAMP(), ?)', $email, AngieApplication::getVisitorUserAgent(), 'failed', AngieApplication::getVisitorIp());
        self::triggerFirewall();
    }

    /**
     * Trigger Firewall.
     *
     * @param User|null $user
     */
    protected static function triggerFirewall($user = null)
    {
        $from_ip = AngieApplication::getVisitorIp();
        $total_attempts = DB::executeFirstCell('SELECT COUNT(*) FROM security_logs WHERE event = ? AND DATE(event_on) = ? AND user_ip = ?', 'failed', DateValue::now()->toMySQL(), $from_ip);

        if ($total_attempts && AngieApplication::firewall()->isEnabled()) {
            $max_attempts = AngieApplication::firewall()->getMaxAttempts();
            $alert_admin_on = AngieApplication::firewall()->getAlertAdminOn();

            if (0 == $total_attempts % $max_attempts) {
                AngieApplication::firewall()->addTempRule($from_ip, $user)->saveConfig();
            }

            // notify admins if has more than X failed logins on day
            if ($total_attempts == $alert_admin_on) {
                AngieApplication::notifications()
                    ->notifyAbout('failed_login')
                    ->setMaxAttempts(AngieApplication::firewall()->getAlertAdminOn())
                    ->setFromIP($from_ip)
                    ->sendToAdministrators(true);
            }
        }
    }
}
