<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace Angie\Middleware;

use AccessLogs;
use ActiveCollab\Authentication\AuthenticatedUser\AuthenticatedUserInterface;
use ActiveCollab\CurrentTimestamp\CurrentTimestampInterface;
use Angie\Middleware\Base\Middleware;
use Angie\Utils\CurrentTimestamp;
use DataManager;
use DataObjectCollection;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Log\LoggerInterface;
use ReflectionClass;
use User;

/**
 * @package Angie\Middleware
 */
class EtagMiddleware extends Middleware
{
    /**
     * @var CurrentTimestampInterface|null
     */
    private $current_timestamp;

    /**
     * EtagMiddleware constructor.
     *
     * @param CurrentTimestampInterface|null $current_timestamp
     * @param LoggerInterface|null           $logger
     */
    public function __construct(CurrentTimestampInterface $current_timestamp = null, LoggerInterface $logger = null)
    {
        parent::__construct($logger);

        $this->current_timestamp = $current_timestamp ? $current_timestamp : new CurrentTimestamp();
    }

    /**
     * {@inheritdoc}
     */
    public function __invoke(ServerRequestInterface $request, ResponseInterface $response, callable $next = null)
    {
        $etag = $this->checkEtag($request->getAttribute('authenticated_user'), trim((string) $request->getHeaderLine('If-None-Match')));

        if ($etag) {
            return $response
                ->withStatus(304)
                ->withHeader('Cache-Control', 'public, max-age=0')
                ->withHeader('Expires', gmdate('D, d M Y H:i:s', ($this->current_timestamp->getCurrentTimestamp() + 315360000)) . ' GMT')
                ->withHeader('Etag', $etag);
        }

        if ($next) {
            $response = $next($request, $response);
        }

        return $response;
    }

    /**
     * Intercept HTTP bootstraping and handle 304 if we have a properly cached resource.
     *
     * @param  AuthenticatedUserInterface|null $user
     * @param  string                          $etag
     * @return bool
     */
    private static function checkEtag(AuthenticatedUserInterface $user = null, $etag)
    {
        if ($etag && substr_count($etag, ',') == 5 && $user instanceof User) {
            list($application_version, $type, $model, $id, $email, $hash) = explode(',', trim($etag, '"'));

            if ($application_version == APPLICATION_VERSION && $user->getEmail() == $email) {
                $etag_ok = false;

                if ($type === 'collection') {
                    if (class_exists($model)) {
                        $model_class = new ReflectionClass($model);

                        if ($model_class->isSubclassOf(DataManager::class)) {
                            $resource = call_user_func("$model::prepareCollection", $id, $user);

                            $etag_ok = $resource instanceof DataObjectCollection && $resource->getTag($user) == $etag;
                        }
                    }
                } elseif ($type === 'object') {
                    $etag_ok = call_user_func("$model::checkObjectEtag", $id, $hash);

                    if ($etag_ok) {
                        AccessLogs::logAccessOnObjectEtagMatch($model, $id, $email);
                    }
                }

                if ($etag_ok) {
                    return $etag;
                }
            }
        }

        return false;
    }
}
