<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace Angie;

use Angie\Search\Adapter;
use Angie\Search\Criterion;
use Angie\Search\Item;
use AngieApplication;
use DataObjectPool;
use DateValue;
use User;

/**
 * @package Angie
 */
final class Search
{
    /**
     * Reset the entire index.
     */
    public static function reset()
    {
        self::getAdapter()->tearDown();
        self::getAdapter()->setUp();

        Events::trigger('on_search_set_up_types');
    }

    /**
     * Query the index and return matching objects.
     *
     * @param  string      $search_for
     * @param  User        $user
     * @param  Criterion[] $criterions
     * @param  int         $page
     * @return array
     */
    public static function query($search_for, User $user, $criterions = null, $page = 1)
    {
        $result = [];

        $search_for = trim($search_for);

        if (empty($search_for)) {
            return $result;
        }

        $records = self::getAdapter()->query($search_for, $user, $criterions, $page);

        if (!empty($records) && is_array($records)) {
            foreach ($records as $record) {
                $object = DataObjectPool::get($record['class'], $record['id']);

                if ($object) {
                    $result[] = [
                        'hit' => $object,
                        'score' => $record['score'],
                        'highlight' => $record['highlight'],
                        'data' => AngieApplication::isInDevelopment() ? $record['data'] : 'IN DEVELOPMENT ONLY',
                    ];
                }
            }
        }

        return $result;
    }

    /**
     * Return suggestions for query searched so far.
     *
     * @param  string $search_for
     * @param  User   $user
     * @return array
     */
    public static function suggest($search_for, User $user)
    {
        return self::getAdapter()->suggest(str_replace_utf('-', ' ', strtolower_utf($search_for)), $user);
    }

    /**
     * Return indexed record for $item (if exists).
     *
     * @param  Item       $item
     * @return array|null
     */
    public static function get(Item $item)
    {
        return self::getAdapter()->get($item);
    }

    /**
     * Add to index.
     *
     * @param Item $item
     * @param bool $bulk
     */
    public static function add(Item $item, $bulk = false)
    {
        self::getAdapter()->add($item, $bulk);
    }

    /**
     * Add to index.
     *
     * @param Item $item
     * @param bool $bulk
     */
    public static function update(Item $item, $bulk = false)
    {
        self::getAdapter()->update($item, $bulk);
    }

    /**
     * Add to index.
     *
     * @param Item $item
     * @param bool $bulk
     */
    public static function remove(Item $item, $bulk = false)
    {
        self::getAdapter()->remove($item, $bulk);
    }

    /**
     * @var Adapter
     */
    private static $adapter;

    /**
     * Return search adapter.
     *
     * @return Adapter
     */
    public static function &getAdapter()
    {
        if (empty(self::$adapter)) {
            self::$adapter = self::loadAdapterInstance(SEARCH_ADAPTER);
        }

        return self::$adapter;
    }

    /**
     * Load and return adapter instance based on adapter name (underscore notation).
     *
     * @param  string       $adapter_name
     * @return Adapter|null
     */
    public static function loadAdapterInstance($adapter_name)
    {
        $class_name = Inflector::camelize($adapter_name);

        $class_file_path = __DIR__ . "/Search/Adapter/$class_name.php";

        if (is_file($class_file_path)) {
            require_once $class_file_path;

            $full_class_name = '\Angie\Search\Adapter\\' . $class_name;

            if (class_exists($full_class_name)) {
                $adapter = new $full_class_name();

                if ($adapter instanceof Adapter) {
                    return $adapter;
                }
            }
        }

        return null;
    }

    /**
     * @var array
     */
    private static $filters = false;

    /**
     * Return fields that can be used to filter the results.
     *
     * Key is field name and value is field type
     *
     * @return array
     */
    public static function getFilters()
    {
        if (self::$filters === false) {
            self::$filters = [];
            Events::trigger('on_search_filters', [&self::$filters]);
        }

        return self::$filters;
    }

    /**
     * Get criterions from request.
     *
     * @param  array      $input
     * @return array|null
     */
    public static function getCriterionsFromRequest($input)
    {
        $result = [];

        if ($input && is_foreachable($input)) {
            foreach (self::getFilters() as $filter => $type) {
                if (empty($input[$filter])) {
                    continue;
                }

                if ($type === Item::FIELD_NUMERIC) {
                    $ids = [];

                    foreach (explode(',', $input[$filter]) as $id) {
                        $id = (int) $id;

                        if ($id) {
                            $ids[] = $id;
                        }
                    }

                    if (count($ids)) {
                        $result[] = new Criterion($filter, $ids);
                    }
                } elseif ($type === Item::FIELD_DATETIME) {
                    if (strpos($input[$filter], ':') === false) {
                        $date = DateValue::makeFromString($input[$filter]);

                        $result[] = new Criterion($filter, [$date->beginningOfDay(), $date->endOfDay()], Criterion::BETWEEN);
                    } else {
                        list($from, $to) = explode(':', $input[$filter]);

                        $result[] = new Criterion($filter, [DateValue::makeFromString($from)->beginningOfDay(), DateValue::makeFromString($to)->endOfDay()], Criterion::BETWEEN);
                    }
                }
            }
        }

        return empty($result) ? null : $result;
    }

    /**
     * Return true if we have a valid filter value.
     *
     * @param  string $field_name
     * @return bool
     */
    public static function isValidFilterField($field_name)
    {
        self::getFilters();

        return isset(self::$filters[$field_name]) && self::$filters[$field_name];
    }

    /**
     * Return true if $value is valid filter value for $field_name.
     *
     * @param  string $field_name
     * @param  mixed  $value
     * @return bool
     * @throws Error
     */
    public static function isValidFilterValue($field_name, $value)
    {
        self::getFilters();

        if (isset(self::$filters[$field_name]) && self::$filters[$field_name]) {
            if (is_array($value) && empty($value)) {
                return false;
            }

            switch (self::$filters[$field_name]) {
                case Item::FIELD_NUMERIC:
                    if (is_array($value)) {
                        foreach ($value as $v) {
                            if (!is_numeric($v)) {
                                return false;
                            }
                        }

                        return true;
                    } else {
                        return is_numeric($value);
                    }

                    break;
                case Item::FIELD_DATETIME:
                    if (is_array($value)) {
                        foreach ($value as $v) {
                            if (!($v instanceof DateValue)) {
                                return false;
                            }
                        }

                        return true;
                    } else {
                        return $value instanceof DateValue;
                    }

                    break;
                default:
                    throw new Error('Currently we support numeric and date time filter columns');
            }
        }

        return false;
    }

    /**
     * Return a list of actions that should be called for index to be rebuilt.
     *
     * @return array
     */
    public static function getRebuildActions()
    {
        $actions = new NamedList();

        Events::trigger('on_search_rebuild_index', [&$actions]);

        return $actions;
    }

    /**
     * Return TRUE if we should index file content.
     *
     * @return bool
     */
    public static function shouldIndexFiles()
    {
        return SEARCH_INDEX_FILES;
    }

    /**
     * Return true if we can index file at $path.
     *
     * @param  string      $path
     * @param  string      $mime_type
     * @param  string|null $name
     * @return bool
     */
    public static function shouldWeIndexThisFile($path, $mime_type = 'application/octet-stream', $name = null)
    {
        return is_file($path) && filesize($path) < self::getMaxFileSizeToIndex();
    }

    /**
     * @return int
     */
    private static function getMaxFileSizeToIndex()
    {
        return SEARCH_MAX_FILE_SIZE_TO_INDEX;
    }

    /**
     * Serialize file for search index.
     *
     * @param  string      $path
     * @param  string      $mime_type
     * @param  string|null $name
     * @return array
     */
    public static function serializeFileForIndex($path, $mime_type = 'application/octet-stream', $name = null)
    {
        return [
            '_content_type' => $mime_type,
            '_name' => $name ? $name : basename($name),
            '_content' => base64_encode(file_get_contents($path)),
        ];
    }

    /**
     * Serialize value for auto-complete.
     *
     * @param  Item         $item
     * @param  array|string $variations
     * @param  string|null  $output_as
     * @param  int|null     $weight
     * @return array
     */
    public static function serializeForAutocomplete(Item $item, $variations, $output_as = null, $weight = null)
    {
        $variations = (array) $variations;

        if (count($variations) === 1) {
            $variations = first($variations);

            if (empty($output_as)) {
                $output_as = $variations;
            }
        } elseif (empty($variations)) {
            $output_as = first($variations);
        }

        $result = [
            'input' => $variations,
            'output' => $output_as,
            'payload' => [
                'type' => $item->getModelName(false, true),
                'id' => $item->getId(),
                'url' => $item->getUrlPath(),
            ],
        ];

        if ($weight) {
            $result['weight'] = $weight;
        }

        return $result;
    }
}
