<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

namespace Angie\Search;

use Angie\Inflector;
use AngieApplication;
use User;

/**
 * Abstract search adapter.
 *
 * @package Angie\Search
 */
abstract class Adapter
{
    /**
     * Set up the index.
     */
    abstract public function setUp();

    /**
     * Destroy the index.
     */
    abstract public function tearDown();

    /**
     * Initialize index for the specific type.
     *
     * @param string|array $type
     * @param array|null   $fields
     */
    abstract public function initializeType($type, $fields = null);

    /**
     * Return indexed record.
     *
     * @param  Item       $item
     * @return array|null
     */
    abstract public function get(Item $item);

    /**
     * Add an item to the index.
     *
     * @param Item $item
     * @param bool $bulk
     */
    abstract public function add(Item $item, $bulk = false);

    /**
     * Update an item.
     *
     * @param Item $item
     * @param bool $bulk
     */
    abstract public function update(Item $item, $bulk = false);

    /**
     * Remove an item.
     *
     * @param Item $item
     * @param bool $bulk
     */
    abstract public function remove(Item $item, $bulk = false);

    /**
     * Query the index.
     *
     * @param  string      $search_for
     * @param  User        $user
     * @param  Criterion[] $criterions
     * @param  int         $page
     * @return array
     */
    abstract public function query($search_for, User $user, $criterions = null, $page = 1);

    /**
     * Suggest the terms for the $search_for.
     *
     * @param  string $search_for
     * @param  User   $user
     * @return array
     */
    abstract public function suggest($search_for, User $user);

    /**
     * Start batch import of the data.
     *
     * @param string $type
     * @param int    $items_per_iteration
     */
    abstract public function batchStart($type, $items_per_iteration = 500);

    /**
     * Add record to the index.
     *
     * @param Item $item
     */
    abstract public function batchAdd($item);

    /**
     * Finished with the batch import.
     *
     * @param string $type
     */
    abstract public function batchDone($type);

    /**
     * Return index name.
     *
     * @return string
     */
    public function getIndexName()
    {
        return ELASTIC_SEARCH_INDEX_NAME ? ELASTIC_SEARCH_INDEX_NAME : Inflector::underscore(AngieApplication::getName()) . '_' . strtolower(APPLICATION_UNIQUE_KEY);
    }

    /**
     * Return number of shareds.
     *
     * @return int
     */
    public function getNumberOfShards()
    {
        return AngieApplication::isOnDemand() ? 1 : 2;
    }

    /**
     * Return number of replicas.
     *
     * @return int
     */
    public function getNumberOfReplicas()
    {
        return AngieApplication::isOnDemand() ? 0 : 1;
    }
}
