<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use Angie\Search;
use Angie\Search\Item as SearchItem;

/**
 * File class.
 *
 * @package ActiveCollab.modules.files
 * @subpackage models
 */
abstract class File extends BaseFile implements IRoutingContext
{
    use IRoutingContextImplementation;

    /**
     * Return routing context name.
     *
     * @return string
     */
    public function getRoutingContext()
    {
        return 'file';
    }

    /**
     * Return routing context parameters.
     *
     * @return mixed
     */
    public function getRoutingContextParams()
    {
        return ['project_id' => $this->getProjectId(), 'file_id' => $this->getId()];
    }

    /**
     * Create a copy of this object and optionally save it.
     *
     * @param  bool $save
     * @return File
     */
    public function copy($save = false)
    {
        /** @var File $copy */
        $copy = parent::copy(false);

        if (is_file($this->getPath())) {
            $copy->setLocation(AngieApplication::storeFile($this->getPath())[1]);
        }

        if ($save) {
            $copy->save();
        }

        return $copy;
    }

    /**
     * Return a list of fields that are indexed for this type.
     *
     * @return array
     */
    public function getSearchFields()
    {
        $result = parent::getSearchFields();

        if (Search::shouldIndexFiles()) {
            $result['content'] = SearchItem::FIELD_ATTACHMENTS;
        }

        return $result;
    }

    /**
     * Serialize object to be indexed.
     *
     * @return array
     */
    public function searchSerialize()
    {
        $result = parent::searchSerialize();

        if (Search::shouldIndexFiles()) {
            if (Search::shouldWeIndexThisFile($this->getPath(), $this->getMimeType(), $this->getName())) {
                $result['content'] = Search::serializeFileForIndex($this->getPath(), $this->getMimeType(), $this->getName());
            }
        }

        return $result;
    }

    /**
     * Return history field renderers.
     *
     * @return array
     */
    public function getHistoryFieldRenderers()
    {
        $renderers = parent::getHistoryFieldRenderers();

        $renderers['is_hidden_from_clients'] = function ($old_value, $new_value, Language $language) {
            if ($new_value) {
                return lang('Marked as hidden from clients', null, true, $language);
            } else {
                return lang('No longer hidden from clients', null, true, $language);
            }
        };

        return $renderers;
    }

    // ---------------------------------------------------
    //  System
    // ---------------------------------------------------

    /**
     * Save to database.
     */
    public function save()
    {
        if ($this->isNew()) {
            $this->setName(Files::getProjectSafeName($this->getName(), $this->getProject()));
        }

        parent::save();
    }
}
