<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use Angie\Http\Request;
use Angie\Http\Response;
use Angie\Http\Response\FileDownload\FileDownload;

AngieApplication::useController('auth_not_required', SystemModule::NAME);

/**
 * Public invoice controller lets people view, export PDF and pay invoices.
 *
 * @package activeCollab.modules.system
 * @subpackage controllers
 */
class PublicInvoiceController extends AuthNotRequiredController
{
    /**
     * Selected invoice.
     *
     * @var Invoice
     */
    protected $active_invoice;

    /**
     * {@inheritdoc}
     */
    protected function __before(Request $request, $user)
    {
        $before_result = parent::__before($request, $user);

        if ($before_result !== null) {
            return $before_result;
        }

        $number = $request->get('number');
        $hash = $request->get('hash');

        if ($number && $hash) {
            $this->active_invoice = Invoices::find(['conditions' => ['number = ? AND hash = ?', $number, $hash], 'one' => true]);
        }

        if (empty($this->active_invoice)) {
            return Response::NOT_FOUND;
        }
    }

    /**
     * @param  Request            $request
     * @return FileDownload|array
     */
    public function view(Request $request)
    {
        if ($request->get('download')) {
            return new FileDownload($this->active_invoice->exportToFile(), 'application/pdf', Invoices::getInvoicePdfName($this->active_invoice));
        }

        if ($recipient = $request->get('recipient')) {
            $values = explode(',', base64_decode($recipient)); // $values = 'john.doe@example.com,John Doe'

            if (count($values) === 2 && is_valid_email($values[0])) {
                if ($user = Users::findByEmail($values[0])) {
                    AccessLogs::logAccess($this->active_invoice, $user);
                } else {
                    AccessLogs::logAccess($this->active_invoice, new AnonymousUser($values[1], $values[0]));
                }
            }
        }

        return $this->getInvoiceAndPaymentFormSettings();
    }

    /**
     * @param  Request     $request
     * @return Invoice|int
     */
    public function make_payment(Request $request)
    {
        /** @var PaymentGateway $gateway */
        $gateway = Payments::getCreditCardGateway($this->active_invoice);

        if ($gateway instanceof PaymentGateway) {
            $amount = $request->put('amount');

            if ($amount < $this->active_invoice->getBalanceDue()) {
                return Response::BAD_REQUEST;
            }

            $this->active_invoice->payWithCreditCard(
                $amount,
                $request->put('name_on_card'),
                $request->put('card_number'),
                $request->put('expiration_month'),
                $request->put('expiration_year'),
                $request->put('security_code')
            );

            return $this->getInvoiceAndPaymentFormSettings(true);
        }

        return Response::NOT_FOUND;
    }

    /**
     * Return invoice and payment settings.
     *
     * @param  bool  $reload_invoice
     * @return array
     */
    private function getInvoiceAndPaymentFormSettings($reload_invoice = false)
    {
        if ($reload_invoice) {
            $this->active_invoice = DataObjectPool::get('Invoice', $this->active_invoice->getId(), true);
        }

        /** @var PaymentGateway $payment_gateway */
        $payment_gateway = Payments::getCreditCardGateway($this->active_invoice);

        $accepted_payment_card = $this->active_invoice->isIssued() && $payment_gateway instanceof PaymentGateway;

        $credit_card_type = $accepted_payment_card ? $this->getCreditCardType($payment_gateway) : null;
        $token = null;
        if ($accepted_payment_card) {
            $token = $payment_gateway->getToken($this->active_invoice);
        }

        return [
            'invoice' => $this->active_invoice,
            'items' => $this->active_invoice->getItems(),
            'template' => new InvoiceTemplate(),
            'payments' => $this->active_invoice->getPayments(),
            'currency' => $this->active_invoice->getCurrency(),
            'creator' => $this->active_invoice->getCreatedBy(),
            'creator_company' => $this->active_invoice->getCreatedBy()->getCompany(),
            'accept_paypal' => $this->active_invoice->isIssued() && Payments::getPayPalGateway($this->active_invoice) instanceof PaymentGateway && Payments::getPayPalGateway($this->active_invoice)->getIsEnabled(),
            'accept_credit_card' => $accepted_payment_card,
            'credit_card_type' => $credit_card_type,
            'credit_card_token' => $token,
            'payflow_form_url' => ($credit_card_type == 'paypaldirect' || $credit_card_type == 'authorize') ? $payment_gateway->getFormUrl() : null,
            'is_currency_conflict' => $credit_card_type == 'paypaldirect' ? $this->active_invoice->getCurrency()->getCode() !== $payment_gateway->getProcessorCurrency() : false,
        ];
    }

    /**
     * Get credit card type.
     *
     * @param  PaymentGateway $payment_gateway
     * @return string
     */
    private function getCreditCardType($payment_gateway)
    {
        return strtolower(str_replace('Gateway', '', $payment_gateway->getType()));
    }
}
