<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

/**
 * Sample invoice for PDF preview.
 *
 * @package activeCollab.modules.invoicing
 * @subpackage models
 */
class SampleInvoice implements IInvoice, IPayments
{
    /**
     * Payments helper.
     *
     * @var ISampleInvoicePaymentsImplementation
     */
    private $payments = false; // getId

    /**
     * Return invoice ID.
     *
     * @return int
     */
    public function getId()
    {
        return 1;
    } // getNumber

    /**
     * Return invoice name.
     *
     * @return string
     */
    public function getName()
    {
        return lang('Invoice #:invoice_num', ['invoice_num' => $this->getNumber()]);
    } // getId

    /**
     * Return invoice number.
     *
     * @return int
     */
    public function getNumber()
    {
        $today = new DateValue();

        return '1/' . $today->getYear();
    } // isIssued

    /**
     * Is invoice issued.
     *
     * @return bool
     */
    public function isIssued()
    {
        return true;
    } // isPaid

    /**
     * Is invoice paid.
     *
     * @return bool
     */
    public function isPaid()
    {
        return false;
    } // isSent

    /**
     * Is invoice sent.
     *
     * @return bool
     */
    public function isSent()
    {
        return false;
    } // isWon

    /**
     * Is invoice issued.
     *
     * @return bool
     */
    public function isWon()
    {
        return false;
    } // isDraft

    /**
     * Is invoice draft.
     *
     * @return bool
     */
    public function isDraft()
    {
        return false;
    } // isCanceled

    /**
     * Is invoice canceled.
     *
     * @return bool
     */
    public function isCanceled()
    {
        return false;
    } // getCreatedOn

    /**
     * Return date when invoice is created.
     *
     * @return DateValue
     */
    public function getCreatedOn()
    {
        return new DateValue();
    } // getDueOn

    /**
     * Return date when invoice is due.
     *
     * @return DateValue
     */
    public function getDueOn()
    {
        return new DateValue('+1 week');
    } // getIssuedOn

    /**
     * Return date when invoice is issued.
     *
     * @return DateValue
     */
    public function getIssuedOn()
    {
        return new DateValue();
    } // getCompanyName

    /**
     * Get company name.
     *
     * @return string
     */
    public function getCompanyName()
    {
        return 'Sample Company Inc.';
    } // getCompanyAddress

    /**
     * Return company address.
     *
     * @return string
     */
    public function getCompanyAddress()
    {
        return "Magic Lane 45\n24325 Illusion District\nUtopia";
    } // getItems

    /**
     * Get paid amount.
     *
     * @return int
     */
    public function getPaidAmount()
    {
        return 0;
    } // getTotal

    /**
     * Get Balance Due.
     *
     * @return flaot
     */
    public function getBalanceDue()
    {
        return $this->getTotal();
    } // getTax

    /**
     * Returned taxed total.
     *
     * @param  bool  $cache
     * @return float
     */
    public function getTotal()
    {
        return $this->getSubTotal() + $this->getTax();
    } // getTotal

    /**
     * Return invoice total.
     *
     * @param  bool  $cache
     * @return float
     */
    public function getSubTotal()
    {
        $total = 0;

        $items = $this->getItems();
        foreach ($items as $item) {
            $total += $item->getQuantity() * $item->getUnitCost();
        } // foreach

        return $total;
    } // getPaidAmount

    /**
     * Get invoice items.
     *
     * @return array
     */
    public function getItems()
    {
        $items = [];

        // find the first tax rate available
        $tax_rate = TaxRates::findOneBySql('SELECT * FROM tax_rates');

        $first_item = new InvoiceItem();
        $first_item->setDescription('Magic Wand');
        $first_item->setQuantity('4');
        $first_item->setUnitCost('20');
        if ($tax_rate instanceof TaxRate) {
            $first_item->setFirstTaxRateId($tax_rate->getId());
        }//if
        $first_item->recalculate();
        $items[] = $first_item;

        $second_item = new InvoiceItem();
        $second_item->setDescription('Box of chocolates');
        $second_item->setQuantity('5');
        $second_item->setUnitCost('10');
        $second_item->recalculate();
        $items[] = $second_item;

        $third_item = new InvoiceItem();
        $third_item->setDescription('White Doves');
        $third_item->setQuantity('2');
        $third_item->setUnitCost('100');
        $third_item->recalculate();
        $items[] = $third_item;

        $fourth_item = new InvoiceItem();
        $fourth_item->setDescription('Invisible Cloak');
        $fourth_item->setQuantity('1');
        $fourth_item->setUnitCost('1000');
        $fourth_item->recalculate();
        $items[] = $fourth_item;

        return $items;
    } // getBalanceDue

    /**
     * Return calculated tax.
     *
     * @param  bool  $cache
     * @return float
     */
    public function getTax()
    {
        $tax_total = 0;

        $items = $this->getItems();
        foreach ($items as $item) {
            $tax_total += $item->getFirstTax() + $item->getSecondTax();
        } // foreach

        return $tax_total;
    } // requireRounding

    /**
     * Require Rounding.
     *
     * @param bool
     */
    public function requireRounding()
    {
        return false;
    } // getLanguage

    /**
     * Returns language of invoice.
     *
     * @param void
     */
    public function getLanguage()
    {
        return null;
    } // getNote

    /**
     * Get invoice note.
     *
     * @param void
     * @return string
     */
    public function getNote()
    {
        $return = "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Sed aliquet ornare est vel ullamcorper. Donec eu leo sed mauris dictum venenatis non in est. Nullam eu feugiat nunc. Nunc pulvinar nibh vitae nisi venenatis vel bibendum libero aliquam.\n\n";
        $return .= "Fusce et turpis aliquam risus tristique porttitor. Nunc in quam at lorem euismod euismod eu ac enim. Fusce venenatis, justo nec pharetra tempor.\n\n";

        return $return;
    } // getCurrency

    /**
     * Get Currency.
     *
     * @return Currency
     */
    public function getCurrency()
    {
        return Currencies::getDefault();
    } // getCurrencyCode

    /**
     * Get currency code.
     *
     * @return string
     */
    public function getCurrencyCode()
    {
        return 'USD';
    } // getSecondTaxIsEnabled

    /**
     * Check if second tax is enabled.
     *
     * @return bool
     */
    public function getSecondTaxIsEnabled()
    {
        return false;
    } // isOverdue

    // ---------------------------------------------------
    //  Interface implementations
    // ---------------------------------------------------

    /**
     * Is sample invoice overdue.
     *
     * @return bool
     */
    public function isOverdue()
    {
        return false;
    }

    /**
     * Return payments helper instance.
     *
     * @return ISampleInvoicePaymentsImplementation
     */
    public function payments()
    {
        if ($this->payments === false) {
            require_once __DIR__ . '/ISampleInvoicePaymentsImplementation.class.php';

            $this->payments = new ISampleInvoicePaymentsImplementation($this);
        } // if

        return $this->payments;
    } // payments
}
