<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

/**
 * Get the list of the new application features.
 *
 * @package ActiveCollab.modules.system
 * @subpackage model
 */
final class NewFeatures
{
    /**
     * Get the list of new features for the user.
     *
     * @param  User   $user
     * @return string
     */
    public static function get(User $user)
    {
        $features = [];
        $all_features = require_once AngieApplication::getModule('system')->getPath() . '/resources/new_features.php';
        $last_visit = self::getLastVisit($user);
        $visibilities = self::getVisibilityScope($user);

        if (!count($all_features)) {
            return [];
        }

        for ($counter = 0; $counter < count($all_features); ++$counter) {
            if (in_array(strtolower($all_features[$counter]['visibility']), $visibilities)) {
                $all_features[$counter]['date'] = strtotime($all_features[$counter]['date']);
                $all_features[$counter]['is_seen'] = $last_visit >= $all_features[$counter]['date'];

                $features[] = $all_features[$counter];
            }
        }

        return $features;
    }

    /**
     * Count new features for user.
     *
     * @param  User $users
     * @return int
     */
    public static function countNew(User $user)
    {
        if (self::areMuted($user)) {
            return 0;
        }

        $count = 0;
        $features = self::get($user);

        if (!count($features)) {
            return 0;
        }

        foreach ($features as $feature) {
            $count += $feature['is_seen'] ? 0 : 1;
        }

        return $count;
    }

    /**
     * Check if new features notifications are muted.
     *
     * @param  User $user
     * @return bool
     */
    public static function areMuted(User $user)
    {
        return !ConfigOptions::getValueFor('new_features_notification', $user);
    }

    /**
     * Get Last Visit.
     *
     * @param  User $user
     * @return int
     */
    public static function getLastVisit(User $user)
    {
        return ConfigOptions::getValueFor('new_features_timestamp', $user);
    }

    /**
     * Record last visit for user.
     *
     * @param  User $user
     * @return bool
     */
    public static function recordLastVisit(User $user)
    {
        ConfigOptions::setValueFor('new_features_timestamp', $user, time());

        return $user->touch();
    }

    /**
     * @param  User  $user
     * @return array
     */
    private static function getVisibilityScope(User $user)
    {
        if ($user->isOwner()) {
            return ['owner', 'manager', 'financial', 'member', 'client'];
        }

        if ($user->isClient()) {
            return ['client'];
        }

        $visibilities = ['member', 'client'];

        if ($user->isFinancialManager()) {
            $visibilities[] = 'financial';
        }

        if ($user->isPowerUser(true)) {
            $visibilities[] = 'manager';
        }

        return $visibilities;
    }
}
