<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use Angie\HTML;

/**
 * @package angie.frameworks.environment
 * @subpackage models
 */
class ZapierWebhookPayloadTransformator extends WebhookPayloadTransformator
{
    /**
     * {@inheritdoc}
     */
    public function shouldTransform($url)
    {
        return strpos($url, 'zapier.com/hooks') !== false;
    }

    /**
     * {@inheritdoc}
     */
    public function transform($event_type, DataObject $payload)
    {
        if (!in_array($event_type, $this->getSupportedEvents())) {
            return null;
        }

        $transformator = $event_type . 'PayloadTransformator';

        if (method_exists(self::class, $transformator)) {
            return $this->$transformator($payload);
        } else {
            throw new Exception("Transformator method {$transformator} not implemented");
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getSupportedEvents()
    {
        return [
            'ProjectCreated',
            'TaskListCreated',
            'TaskCreated',
            'CommentCreated',
            'TimeRecordCreated',
            'TaskCompleted',
            'TaskListChanged',
        ];
    }

    /**
     * Transform payload when project is created.
     *
     * @param  Project $project
     * @return array
     */
    public function ProjectCreatedPayloadTransformator(Project $project)
    {
        return [
            'id' => $project->getId(),
            'name' => $project->getName(),
            'created_by_id' => $project->getCreatedById(),
            'created_by_name' => $project->getCreatedByName(),
        ];
    }

    /**
     * Transform payload when task list is created.
     *
     * @param  TaskList $task_list
     * @return array
     */
    public function TaskListCreatedPayloadTransformator(TaskList $task_list)
    {
        return [
            'id' => $task_list->getId(),
            'name' => $task_list->getName(),
            'project_id' => $task_list->getProjectId(),
            'project_name' => $task_list->getProject()->getName(),
        ];
    }

    /**
     * Transform payload when task is created.
     *
     * @param  Task  $task
     * @return array
     */
    public function TaskCreatedPayloadTransformator(Task $task)
    {
        return [
            'id' => $task->getId(),
            'name' => $task->getName(),
            'body' => HTML::toPlainText($task->getBody()),
            'project_id' => $task->getProjectId(),
            'project_name' => $task->getProject()->getName(),
            'task_list_id' => $task->getTaskListId(),
            'task_list_name' => $task->getTaskList()->getName(),
        ];
    }

    /**
     * Transform payload when comment is created.
     *
     * @param  Comment $comment
     * @return array
     */
    public function CommentCreatedPayloadTransformator(Comment $comment)
    {
        return [
            'id' => $comment->getId(),
            'body' => HTML::toPlainText($comment->getBody()),
            'parent_type' => $comment->getParentType(),
            'parent_id' => $comment->getParentId(),
            'parent_name' => $comment->getParent()->getName(),
            'created_by_id' => $comment->getCreatedById(),
            'created_by_name' => $comment->getCreatedByName(),
        ];
    }

    /**
     * Transform payload when time record is created.
     *
     * @param  TimeRecord $time_record
     * @return array
     */
    public function TimeRecordCreatedPayloadTransformator(TimeRecord $time_record)
    {
        return [
            'id' => $time_record->getId(),
            'parent_type' => $time_record->getParentType(),
            'parent_id' => $time_record->getParentId(),
            'parent_name' => $time_record->getParent()->getName(),
            'value' => $time_record->getValue(),
            'job_type_id' => $time_record->getJobTypeId(),
            'created_by_id' => $time_record->getCreatedById(),
            'created_by_name' => $time_record->getCreatedByName(),
            'billable_status' => $time_record->getBillableStatus(),
            'description' => $time_record->getSummary(),
        ];
    }

    /**
     * Transform payload when task is completed.
     *
     * @param  Task  $task
     * @return array
     */
    public function TaskCompletedPayloadTransformator(Task $task)
    {
        return [
            'id' => $task->getId(),
            'name' => $task->getName(),
            'project_id' => $task->getProjectId(),
            'project_name' => $task->getProject()->getName(),
            'task_list_id' => $task->getTaskListId(),
            'task_list_name' => $task->getTaskList()->getName(),
            'completed_by_id' => $task->getCompletedBy()->getId(),
            'completed_by_name' => $task->getCompletedBy()->getName(),
        ];
    }

    /**
     * Transform payload when change task list.
     *
     * @param  Task  $task
     * @return array
     */
    public function TaskListChangedPayloadTransformator(Task $task)
    {
        return [
            'id' => $task->getId(),
            'name' => $task->getName(),
            'project_id' => $task->getProjectId(),
            'project_name' => $task->getProject()->getName(),
            'task_list_id' => $task->getTaskListId(),
            'task_list_name' => $task->getTaskList()->getName(),
        ];
    }
}
