<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

/**
 * Project tasks collection.
 *
 * @package ActiveCollab.modules.task
 * @subpackage models
 */
class ProjectTasksCollection extends CompositeCollection
{
    use IWhosAsking;

    /**
     * @var Project
     */
    private $project;
    /**
     * Cached tag value.
     *
     * @var string
     */
    private $tag = false;

    /**
     * @var string
     */
    private $timestamp_hash = false;

    /**
     * @var ModelCollection
     */
    private $tasks_collection = false, $task_lists_collection = false;

    /**
     * @var int[]
     */
    private $task_ids = false;

    /**
     * @return string
     */
    public function getModelName()
    {
        return 'Tasks';
    }

    /**
     * @param  Project $project
     * @return $this
     */
    public function &setProject(Project $project)
    {
        $this->project = $project;

        return $this;
    }

    /**
     * Return collection etag.
     *
     * @param  IUser  $user
     * @param  bool   $use_cache
     * @return string
     */
    public function getTag(IUser $user, $use_cache = true)
    {
        if ($this->tag === false || empty($use_cache)) {
            $this->tag = $this->prepareTagFromBits($user->getEmail(), $this->getTimestampHash());
        }

        return $this->tag;
    }

    /**
     * @return string
     */
    private function getTimestampHash()
    {
        if ($this->timestamp_hash === false) {
            $this->timestamp_hash = sha1(
                $this->getTasksCollection()->getTimestampHash('updated_on') . '-' .
                $this->getTaskListsCollection()->getTimestampHash('updated_on') . '-' .
                $this->project->getUpdatedOn()->toMySQL()
            );
        }

        return $this->timestamp_hash;
    }

    /**
     * @return ModelCollection
     */
    private function getTasksCollection()
    {
        if ($this->tasks_collection === false) {
            $this->tasks_collection = Tasks::prepareCollection('active_tasks_in_project_' . $this->project->getId(), $this->getWhosAsking());
        }

        return $this->tasks_collection;
    }

    /**
     * @return ModelCollection
     */
    private function getTaskListsCollection()
    {
        if ($this->task_lists_collection === false) {
            $this->task_lists_collection = TaskLists::prepareCollection('all_task_lists_in_project_' . $this->project->getId(), $this->getWhosAsking());
        }

        return $this->task_lists_collection;
    }

    /**
     * @return array
     */
    public function execute()
    {
        if ($task_ids = $this->getTaskIds()) {
            Comments::preloadCountByParents('Task', $task_ids);
            Subtasks::preloadCountByTasks($task_ids);
            Attachments::preloadDetailsByParents('Task', $task_ids);
            Labels::preloadDetailsByParents('Task', $task_ids);
        }

        return [
            'tasks' => $this->getTasksCollection(),
            'task_lists' => $this->getTaskListsCollection(),
            'label_ids' => Labels::getLabelIdsByProject($this->project),
            'project' => $this->project,
            'completed_tasks_count' => Tasks::countCompletedByProject($this->project),
        ];
    }

    /**
     * Return list of task ID-s that are captured by this collection.
     *
     * @return array|null
     */
    private function getTaskIds()
    {
        if ($this->task_ids === false) {
            $this->task_ids = DB::executeFirstColumn('SELECT id FROM tasks WHERE project_id = ? AND completed_on IS NULL AND is_trashed = ?', $this->project->getId(), false);
        }

        return $this->task_ids;
    }

    /**
     * @return int
     */
    public function count()
    {
        if ($task_ids = $this->getTaskIds()) {
            return count($task_ids);
        }

        return 0;
    }
}
