<?php

/*
 * This file is part of the Active Collab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use Angie\Http\Request;
use Angie\Http\Response;

AngieApplication::useController('project', SystemModule::NAME);

/**
 * Expenses controller.
 *
 * @package activeCollab.modules.tracking
 * @subpackage controllers
 */
final class ExpensesController extends ProjectController
{
    /**
     * Selected expense instance.
     *
     * @var Expense
     */
    protected $active_expense;

    /**
     * {@inheritdoc}
     */
    protected function __before(Request $request, $user)
    {
        if ($response = parent::__before($request, $user)) {
            return $response;
        }

        $this->active_expense = DataObjectPool::get('Expense', $request->getId('expense_id'));

        if (empty($this->active_expense)) {
            $this->active_expense = new Expense();
            $this->active_expense->setParent($this->active_project);
        }

        if (!$this->active_expense->getProject()->is($this->active_project)) {
            return Response::NOT_FOUND;
        }

        return null;
    }

    /**
     * List project expenses.
     *
     * @param  Request              $request
     * @param  User                 $user
     * @return ModelCollection|void
     */
    public function index(Request $request, User $user)
    {
        if ($user instanceof Client && !$this->active_project->getIsClientReportingEnabled()) {
            return Response::NOT_FOUND;
        }

        AccessLogs::logAccess($this->active_project, $user);

        return Expenses::prepareCollection('expenses_in_project_' . $this->active_project->getId() . '_page_' . $request->getPage(), $user);
    }

    /**
     * Log a new expense.
     *
     * @param  Request             $request
     * @param  User                $user
     * @return DataObject|int|void
     */
    public function add(Request $request, User $user)
    {
        $post = $request->post();

        if (isset($post['task_id']) && $post['task_id']) {
            $track_expense_for = DataObjectPool::get('Task', $post['task_id']);
        }

        if (empty($post['user_id']) || !Expenses::canTrackForOthers($user, $this->active_project)) {
            $post['user_id'] = $user->getId(); // Force user to be logged user unless sender is owner or project manager
        }

        if (empty($track_expense_for)) {
            $track_expense_for = $this->active_project;
        }

        if ($track_expense_for->canTrackExpenses($user)) {
            $post['parent_type'] = get_class($track_expense_for);
            $post['parent_id'] = $track_expense_for->getId();

            if (!array_key_exists('billable_status', $post)) {
                $post['billable_status'] = $this->active_project->getDefaultBillableStatus();
            }

            return Expenses::create($post);
        } else {
            return Response::NOT_FOUND;
        }
    }

    /**
     * Show expense data.
     *
     * @param  Request     $request
     * @param  User        $user
     * @return int|Expense
     */
    public function view(Request $request, User $user)
    {
        return $this->active_expense->isLoaded() && $this->active_expense->canView($user) ? $this->active_expense : Response::NOT_FOUND;
    }

    /**
     * Update a selected expense.
     *
     * @param  Request        $request
     * @param  User           $user
     * @return DataObject|int
     */
    public function edit(Request $request, User $user)
    {
        if ($this->active_expense->isLoaded() && $this->active_expense->canEdit($user)) {
            $put = $request->put();

            if (array_key_exists('user_id', $put)) {
                $new_user_id = (int) $put['user_id'];

                if ($new_user_id > 0) {
                    if ($new_user_id != $this->active_expense->getUserId() && !Expenses::canTrackForOthers($user, $this->active_project)) {
                        return Response::FORBIDDEN;
                    }

                    $put['user_id'] = $new_user_id;
                } else {
                    return Response::BAD_REQUEST;
                }
            }

            foreach (['parent_type', 'parent_id', 'task_id', 'project_id'] as $k) {
                if (array_key_exists($k, $put)) {
                    unset($put[$k]);
                }
            }

            return Expenses::update($this->active_expense, $put);
        }

        return Response::NOT_FOUND;
    }

    /**
     * @param  Request              $request
     * @param  User                 $user
     * @return Expense|int
     * @throws InvalidInstanceError
     */
    public function move(Request $request, User $user)
    {
        if ($this->active_expense->isLoaded() && $this->active_expense->canEdit($user)) {
            $move_to = $this->getMoveToParentFromPut($request->put());

            if ($move_to instanceof ITracking) {
                $this->active_expense->setParent($move_to, true);
            } else {
                return Response::BAD_REQUEST;
            }

            return $this->active_expense;
        }

        return Response::NOT_FOUND;
    }

    /**
     * Return target task or project from PUT parameters.
     *
     * @param  array             $put
     * @return Project|Task|null
     */
    private function getMoveToParentFromPut($put)
    {
        if (array_key_exists('task_id', $put)) {
            if ($put['task_id']) {
                return DataObjectPool::get('Task', $put['task_id']);
            }

            return DataObjectPool::get('Project', (isset($put['project_id']) ? $put['project_id'] : null));
        }

        if (array_key_exists('project_id', $put)) {
            return DataObjectPool::get('Project', $put['project_id']);
        }

        return null;
    }

    /**
     * Move selected expense to trash.
     *
     * @param  Request             $request
     * @param  User                $user
     * @return bool|DataObject|int
     */
    public function delete(Request $request, User $user)
    {
        return $this->active_expense->isLoaded() && $this->active_expense->canDelete($user) ? Expenses::scrap($this->active_expense) : Response::NOT_FOUND;
    }
}
