<?php
/**
 * Plugin Helper File
 *
 * @package         Tooltips
 * @version         3.7.11
 *
 * @author          Peter van Westen <peter@nonumber.nl>
 * @link            http://www.nonumber.nl
 * @copyright       Copyright © 2015 NoNumber All Rights Reserved
 * @license         http://www.gnu.org/licenses/gpl-2.0.html GNU/GPL
 */

defined('_JEXEC') or die;

// Load common functions
require_once JPATH_PLUGINS . '/system/nnframework/helpers/functions.php';
require_once JPATH_PLUGINS . '/system/nnframework/helpers/text.php';
require_once JPATH_PLUGINS . '/system/nnframework/helpers/protect.php';

nnFrameworkFunctions::loadLanguage('plg_system_tooltips');

/**
 * Plugin that replaces stuff
 */
class plgSystemTooltipsHelper
{
	public function __construct(&$params)
	{
		$this->params = $params;

		$this->params->comment_start = '<!-- START: Tooltips -->';
		$this->params->comment_end = '<!-- END: Tooltips -->';

		$this->params->tag = preg_replace('#[^a-z0-9-_]#s', '', $this->params->tag);
		$this->params->regex = '#'
			. '\{' . preg_quote($this->params->tag, '#') . '((?:\s|&nbsp;|&\#160;|<)(?:[^\}]*\{[^\}]*\})*[^\}]*)\}'
			. '(.*?)'
			. '\{/' . preg_quote($this->params->tag, '#') . '\}'
			. '#s';

		$this->params->protected_tags = array(
			$this->params->tag
		);

		$this->params->disabled_components = array('com_acymailing');
	}

	public function onContentPrepare(&$article, &$context)
	{
		nnFrameworkHelper::processArticle($article, $context, $this, 'replaceTags');
	}

	public function onAfterDispatch()
	{
		// only in html
		if (JFactory::getDocument()->getType() !== 'html' && JFactory::getDocument()->getType() !== 'feed')
		{
			return;
		}

		$buffer = JFactory::getDocument()->getBuffer('component');

		if (empty($buffer) || is_array($buffer))
		{
			return;
		}

		// do not load scripts/styles on print page
		if (JFactory::getDocument()->getType() !== 'feed' && !JFactory::getApplication()->input->getInt('print', 0) && !JFactory::getApplication()->input->getInt('noscript', 0))
		{
			if ($this->params->load_mootools)
			{
				JHtml::_('behavior.mootools');
			}

			JFactory::getDocument()->addScriptDeclaration('/* START: Tooltips scripts */');
			$script = '
				var tooltips_max_width = ' . (int) $this->params->max_width . ';
				var tooltips_fade_in_speed = 0;
				var tooltips_fade_out_speed = 200;
			';
			JFactory::getDocument()->addScriptDeclaration(preg_replace('#\n\s*#s', ' ', trim($script)));

			$options = array();
			$options['maxTitleChars'] = 99999;
			$options['className'] = 'tooltips-tip';
			$options['onShow'] = "function(tip){ tooltips_show( tip ); }";
			$options['onHide'] = "function(tip){ tooltips_hide( tip ); }";

			JHtml::_('behavior.tooltip', '.tooltips-link', $options);
			$options['className'] = 'tooltips-img tooltips-tip';
			JHtml::_('behavior.tooltip', '.tooltips-link-img', $options);

			JFactory::getDocument()->addScriptDeclaration('/* END: Tooltips scripts */');

			JHtml::script('tooltips/script.min.js', false, true);
			if ($this->params->load_stylesheet)
			{
				JHtml::stylesheet('tooltips/style.min.css', false, true);
			}

			$style = array();
			if ($this->params->underline)
			{
				$style[] = 'span.tooltips-link, span.tooltips-link-img { border-bottom: 1px ' . $this->params->underline . ' #' . $this->params->underline_color . '; }';
			}
			if ($this->params->zindex)
			{
				$style[] = 'div.tooltips-tip, div.tooltips-tip.tool-tip, div.tooltips-tip-tip { z-index: ' . (int) $this->params->zindex . ' }';
			}
			$style[] = 'div.tooltips-tip div.tip, div.tooltips-tip-tip > div {';
			if ($this->params->text_color)
			{
				$style[] = 'color: #' . strtoupper($this->params->text_color) . ';';
			}
			$style[] = 'background-color: #' . strtoupper($this->params->bg_color) . ';';
			$style[] = 'border-color: #' . strtoupper($this->params->border_color) . ';';
			$style[] = 'border-width: ' . (int) $this->params->border_width . 'px;';
			if (!$this->params->use_border_radius)
			{
				$style[] = '-webkit-border-radius: 0;';
				$style[] = '-moz-border-radius: 0;';
				$style[] = 'border-radius: 0;';
			}
			else if ($this->params->border_radius != 10)
			{
				$style[] = '-webkit-border-radius: ' . (int) $this->params->border_radius . 'px;';
				$style[] = '-moz-border-radius: ' . (int) $this->params->border_radius . 'px;';
				$style[] = 'border-radius: ' . (int) $this->params->border_radius . 'px;';
			}
			$style[] = 'max-width: ' . (int) $this->params->max_width . 'px;';
			$style[] = '}';
			if ($this->params->link_color)
			{
				$style[] = 'div.tooltips-tip div.tip a, div.tooltips-tip-tip > div a {';
				$style[] = 'color: #' . strtoupper($this->params->link_color) . ';';
				$style[] = '}';
			}
			JFactory::getDocument()->addStyleDeclaration('/* START: Tooltips styles */ ' . implode(' ', $style) . ' /* END: Tooltips styles */');
		}

		if (strpos($buffer, '{' . $this->params->tag) === false)
		{
			return;
		}

		$this->replaceTags($buffer, 'component');

		JFactory::getDocument()->setBuffer($buffer, 'component');
	}

	public function onAfterRender()
	{
		// only in html and feeds
		if (JFactory::getDocument()->getType() !== 'html' && JFactory::getDocument()->getType() !== 'feed')
		{
			return;
		}

		$html = JResponse::getBody();
		if ($html == '')
		{
			return;
		}

		if (strpos($html, '{' . $this->params->tag) === false)
		{
			if (strpos($html, 'class="tooltips-link') === false)
			{
				// remove style and script if no items are found
				$html = preg_replace('#\s*<' . 'link [^>]*href="[^"]*/(tooltips/css|css/tooltips)/[^"]*\.css[^"]*"[^>]* />#s', '', $html);
				$html = preg_replace('#\s*<' . 'script [^>]*src="[^"]*/(tooltips/js|js/tooltips)/[^"]*\.js[^"]*"[^>]*></script>#s', '', $html);
				$html = preg_replace('#/\* START: Tooltips .*?/\* END: Tooltips [a-z]* \*/\s*#s', '', $html);
			}

			$this->cleanLeftoverJunk($html);

			JResponse::setBody($html);

			return;
		}

		// only do stuff in body
		list($pre, $body, $post) = nnText::getBody($html);
		$this->replaceTags($body, 'body');
		$html = $pre . $body . $post;

		$this->cleanLeftoverJunk($html);

		JResponse::setBody($html);
	}

	public function replaceTags(&$string, $area = 'article')
	{
		if (!is_string($string) || $string == '')
		{
			return;
		}

		// allow in component?
		if ($area == 'component'
			&& in_array(JFactory::getApplication()->input->get('option'), $this->params->disabled_components)
		)
		{
			$this->protectTags($string);

			return;
		}

		if (strpos($string, '{' . $this->params->tag) === false)
		{
			return;
		}

		$this->protect($string);

		if (preg_match_all($this->params->regex, $string, $matches, PREG_SET_ORDER) > 0)
		{
			foreach ($matches as $match)
			{
				$tip = $match['1'];
				$text = $match['2'];

				$classes = str_replace('\|', '[:TT_BAR:]', $tip);
				$classes = explode('|', $classes);
				foreach ($classes as $i => $class)
				{
					$classes[$i] = trim(str_replace('[:TT_BAR:]', '|', $class));
				}
				$tip = array_shift($classes);
				array_unshift($classes, 'tooltips-link');

				if (preg_match_all('#href="([^"]*)"#si', $tip, $url_matches, PREG_SET_ORDER) > 0)
				{
					foreach ($url_matches as $url_match)
					{
						$url = 'href="' . JRoute::_($url_match['1']) . '"';
						$tip = str_replace($url_match['0'], $url, $tip);
					}
				}
				if (preg_match_all('#src="([^"]*)"#si', $tip, $url_matches, PREG_SET_ORDER) > 0)
				{
					foreach ($url_matches as $url_match)
					{
						$url = $url_match['1'];
						if (strpos($url, 'http') !== 0)
						{
							$url = JURI::root() . $url;
						}
						$url = 'src="' . $url . '"';
						$tip = str_replace($url_match['0'], $url, $tip);
					}
				}
				$tip = explode('::', $tip, 2);
				if (!isset($tip['1']))
				{
					if (preg_match('#^\s*(&lt;|<)img [^>]*(&gt;|>)\s*$#', $tip['0']))
					{
						$classes['0'] .= '-img';
						$tip['1'] = ' ';
					}
					else
					{
						$tip['1'] = $tip['0'];
						$tip['0'] = '';
					}
				}
				$tip = implode('::', $tip);

				if (preg_match('#^\s*<img [^>]*>\s*$#', $text))
				{
					$classes[] = 'isimg';
				}

				$this->fixUrls($tip);

				$r = '<span'
					. ' class="' . implode(' ', $classes) . '"'
					. ' title="' . $this->makeSave($tip) . '">' . $text . '</span>';
				$string = str_replace($match['0'], $r, $string);
			}
		}

		nnProtect::unprotect($string);
	}

	private function fixUrls(&$string)
	{
		if (preg_match_all('#(?:src|href)="([^"]+)"#s', $string, $matches, PREG_SET_ORDER) < 1)
		{
			return;
		}

		foreach ($matches as $match)
		{
			$url = $match['1'];

			if (
				$this->isExternal($url)
				|| substr($url, 0, 4) == 'http'
			)
			{
				continue;
			}

			$fixed_url = JRoute::_(JUri::base() . ltrim($url, '/'));

			$string = str_replace(
				$match['0'],
				str_replace($url, $fixed_url, $match['0']),
				$string
			);
		}
	}

	private function isExternal($url)
	{
		if (strpos($url, '://') === false)
		{
			return 0;
		}

		// hostname: give preference to SERVER_NAME, because this includes subdomains
		$hostname = ($_SERVER['SERVER_NAME']) ? $_SERVER['SERVER_NAME'] : $_SERVER['HTTP_HOST'];

		return !(strpos(preg_replace('#^.*?://#', '', $url), $hostname) === 0);
	}

	private function makeSave($string)
	{
		$string = preg_replace('#^\s*&lt;(img .*)&gt;\s*$#', '<\1>', $string);

		return str_replace(array('&', '"', '<', '>'), array('&amp;', '&quot;', '&lt;', '&gt;'), $string);
	}

	private function protect(&$string)
	{
		nnProtect::protectFields($string);
		nnProtect::protectSourcerer($string);
	}

	private function protectTags(&$string)
	{
		nnProtect::protectTags($string, $this->params->protected_tags);
	}

	private function unprotectTags(&$string)
	{
		nnProtect::unprotectTags($string, $this->params->protected_tags);
	}

	/**
	 * Just in case you can't figure the method name out: this cleans the left-over junk
	 */
	private function cleanLeftoverJunk(&$string)
	{
		$this->unprotectTags($string);

		nnProtect::removeInlineComments($string, 'Tooltips');
	}
}
